package org.criticalsection.maven.scm.provider.monotone.command.checkout;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.command.checkout.AbstractCheckOutCommand;
import org.apache.maven.scm.command.checkout.CheckOutScmResult;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmVersion;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.codehaus.plexus.util.FileUtils;
import org.codehaus.plexus.util.StringUtils;
import org.criticalsection.mtn4java.commands.Command;
import org.criticalsection.mtn4java.commands.standalone.CheckoutCommand;
import org.criticalsection.mtn4java.commands.invokers.Invoker;
import org.criticalsection.mtn4java.commands.invokers.StandaloneInvoker;
import org.criticalsection.mtn4java.commands.invokers.InvokationException;

import java.io.File;
import java.io.IOException;

import org.criticalsection.maven.scm.provider.monotone.MonotoneCommandlineUtils;
import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;

public class MonotoneCheckOutCommand
    extends AbstractCheckOutCommand
{

    //-----------------------------------------------------------------------------------------
    //  AbstractCheckOutCommand implementation
    //-----------------------------------------------------------------------------------------

    protected CheckOutScmResult executeCheckOutCommand( ScmProviderRepository repository, ScmFileSet fileSet, ScmVersion version )
        throws ScmException
    {
        getLogger().debug( "Executing checkout command" );

        //-----------------------------------------------------------------------------------------
        //  Precondition checks
        //-----------------------------------------------------------------------------------------

        File checkoutDir = fileSet.getBasedir();

        if ( checkoutDir.exists() )
        {
            try
            {
                FileUtils.deleteDirectory( checkoutDir );
            }
            catch ( IOException e )
            {
                throw new ScmException( "The checkout dir already exists and I can't delete it.", e );
            }
        }

        //-----------------------------------------------------------------------------------------
        //  checkout working copy
        //-----------------------------------------------------------------------------------------

        Command cmd = createCommand( repository, checkoutDir, version );

        try
        {
            Invoker invoker = new StandaloneInvoker( "mtn",fileSet.getBasedir().getParentFile().getAbsolutePath() );
            invoker.invoke( cmd );
        }
        catch ( InvokationException e )
        {
            if ( e.getType() == InvokationException.Type.PRECONDITION_ERROR )
            {
                return new CheckOutScmResult( "No commandline cvailable", "Precondition error during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.PROCESS_CREATION_ERROR )
            {
                return new CheckOutScmResult( e.getCommandline(), "Error while crreating the external process during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.EXECUTION_ERROR )
            {
                return new CheckOutScmResult( e.getCommandline(), "Error during command execution.", e.getFormatedErrorMessage(), false );
            }
        }

        return new CheckOutScmResult( MonotoneCommandlineUtils.getWorkingCopyFiles( checkoutDir ), new ScmResult( null, null, null, true ) );
    }

    //-----------------------------------------------------------------------------------------
    //  MonotoneCheckOutCommand implementation
    //-----------------------------------------------------------------------------------------

    Command createCommand( ScmProviderRepository repository, File checkoutDirectory, ScmVersion version )
        throws ScmRepositoryException
    {
        MonotoneScmProviderRepository repo = ((MonotoneScmProviderRepository)repository);

        // Create base command
        CheckoutCommand cmd = new CheckoutCommand();

        // Set base options
        if ( repo.getConfigDirectory() != null && StringUtils.isNotEmpty( repo.getConfigDirectory() ) )
        {
            cmd.setConfigDirPath( repo.getConfigDirectory() );
        }

        if ( repo.getKeyringDirectory() != null && StringUtils.isNotEmpty( repo.getKeyringDirectory() ) )
        {
            cmd.setKeyDirPath( repo.getKeyringDirectory() );
        }

        if ( repo.getKey() != null && StringUtils.isNotEmpty( repo.getKey() ) )
        {
            cmd.setKey( repo.getKey() );
        }

        // Add database
        cmd.setDatabasePath( repo.getDatabase() );

        // Add bracnch/revision selector
        if ( version != null && version.getType().equals( "Branch" ) )
        {
            cmd.setBranch( version.getName() );
        }
        else if ( version != null && version.getType().equals( "Revision" ) )
        {
            cmd.setRevision( "i:" + version.getName() );
        }
        else if ( version != null && version.getType().equals( "Tag" ) )
        {
            cmd.setRevision( "t:" + version.getName() );
        }
        else if ( repo.getBranch() != null )
        {
            cmd.setBranch( repo.getBranch() );
        }
        else if ( repo.getTag() != null )
        {
            cmd.setRevision( "t:" + repo.getTag() );
        }
        else
        {
            throw new ScmRepositoryException( "No revision to check out from specified." );
        }

        // Add checkout directory
        cmd.setWorkspacePath( checkoutDirectory.getAbsolutePath() );

        return cmd;
    }
}
