package org.criticalsection.maven.scm.provider.monotone.command.checkin;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.command.checkin.AbstractCheckInCommand;
import org.apache.maven.scm.command.checkin.CheckInScmResult;
import org.apache.maven.scm.command.status.StatusScmResult;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmVersion;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.FileUtils;
import org.criticalsection.mtn4java.commands.Command;
import org.criticalsection.mtn4java.commands.standalone.CommitCommand;
import org.criticalsection.mtn4java.commands.invokers.Invoker;
import org.criticalsection.mtn4java.commands.invokers.StandaloneInvoker;
import org.criticalsection.mtn4java.commands.invokers.InvokationException;
import org.criticalsection.maven.scm.provider.monotone.command.status.MonotoneStatusCommand;
import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;

import java.io.File;
import java.io.IOException;

public class MonotoneCheckInCommand
    extends AbstractCheckInCommand
{

    protected CheckInScmResult executeCheckInCommand( ScmProviderRepository repository, ScmFileSet fileSet, String message, ScmVersion scmVersion )
        throws ScmException
    {
        getLogger().debug( "Executing checkin command" );

        //-----------------------------------------------------------------------------------------
        //  Precondition checks
        //-----------------------------------------------------------------------------------------

        if ( scmVersion != null )
        {
            throw new ScmException( "This provider can't handle tags." );
        }

        if ( fileSet.getFileList().size() != 0 )
        {
            getLogger().warn( "This provider doesn't support committing individual files. Commiting whole workspace." );
        }

        //-----------------------------------------------------------------------------------------
        //  check the workspapce status
        //-----------------------------------------------------------------------------------------

        // get the changed files
        MonotoneStatusCommand statusCmd = new MonotoneStatusCommand();
        statusCmd.setLogger( getLogger() );
        StatusScmResult statusResult = statusCmd.executeStatusCommand( repository, fileSet );

        if ( !statusResult.isSuccess() )
        {
            return new CheckInScmResult( statusResult.getCommandLine(), "Error obtaining the status of the working copy.", statusResult.getCommandOutput(), false );
        }

        //-----------------------------------------------------------------------------------------
        //  commit working copy
        //-----------------------------------------------------------------------------------------

        File messageFile = FileUtils.createTempFile( "maven-scm-", ".commit", null );

        try
        {
            FileUtils.fileWrite( messageFile.getAbsolutePath(), message );
        }
        catch ( IOException ex )
        {
            return new CheckInScmResult( null, "Error while making a temporary file for the commit message: " +
                ex.getMessage(), null, false );
        }

        Command cmd = createCommand( repository, messageFile.getAbsolutePath() );

        try
        {
            Invoker invoker = new StandaloneInvoker( "mtn",fileSet.getBasedir().getAbsolutePath() );
            invoker.invoke( cmd );
        }
        catch ( InvokationException e )
        {
            if ( e.getType() == InvokationException.Type.PRECONDITION_ERROR )
            {
                return new CheckInScmResult( "No commandline cvailable", "Precondition error during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.PROCESS_CREATION_ERROR )
            {
                return new CheckInScmResult( e.getCommandline(), "Error while crreating the external process during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.EXECUTION_ERROR )
            {
                return new CheckInScmResult( e.getCommandline(), "Error during command execution.", e.getFormatedErrorMessage(), false );
            }
        }
        finally
        {
            try
            {
                FileUtils.forceDelete( messageFile );
            }
            catch ( IOException ex )
            {
                // ignore
            }
        }

        return new CheckInScmResult( statusResult.getChangedFiles(), new ScmResult( null, null, null, true ) );
    }

    //-----------------------------------------------------------------------------------------
    //  MonotoneCheckInCommand implementation
    //-----------------------------------------------------------------------------------------

    Command createCommand( ScmProviderRepository repository, String message )
        throws ScmRepositoryException
    {
        MonotoneScmProviderRepository repo = ((MonotoneScmProviderRepository)repository);

        // Create base command
        CommitCommand cmd = new CommitCommand();

        // Set base options
        if ( repo.getConfigDirectory() != null && StringUtils.isNotEmpty( repo.getConfigDirectory() ) )
        {
            cmd.setConfigDirPath( repo.getConfigDirectory() );
        }

        if ( repo.getKeyringDirectory() != null && StringUtils.isNotEmpty( repo.getKeyringDirectory() ) )
        {
            cmd.setKeyDirPath( repo.getKeyringDirectory() );
        }

        if ( repo.getKey() != null && StringUtils.isNotEmpty( repo.getKey() ) )
        {
            cmd.setKey( repo.getKey() );
        }

        cmd.setMessageFilePath( message );

        return cmd;
    }
}
