package org.criticalsection.maven.scm.provider.monotone.command.branch;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.command.branch.AbstractBranchCommand;
import org.apache.maven.scm.command.branch.BranchScmResult;
import org.apache.maven.scm.command.status.StatusScmResult;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.codehaus.plexus.util.StringUtils;
import org.criticalsection.mtn4java.commands.Command;
import org.criticalsection.mtn4java.commands.standalone.CertCommand;
import org.criticalsection.mtn4java.commands.invokers.Invoker;
import org.criticalsection.mtn4java.commands.invokers.StandaloneInvoker;
import org.criticalsection.mtn4java.commands.invokers.InvokationException;

import org.criticalsection.maven.scm.provider.monotone.MonotoneCommandlineUtils;
import org.criticalsection.maven.scm.provider.monotone.command.status.MonotoneStatusCommand;
import org.criticalsection.maven.scm.provider.monotone.command.baserevision.MonotoneBaseRevisionCommand;
import org.criticalsection.maven.scm.provider.monotone.command.baserevision.BaseRevisionScmResult;
import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;

public class MonotoneBranchCommand extends AbstractBranchCommand
{
    protected ScmResult executeBranchCommand( ScmProviderRepository repository, ScmFileSet fileSet, String branchName,
                                              String message )
        throws ScmException
    {
        //-----------------------------------------------------------------------------------------
        //  Precondition checks
        //-----------------------------------------------------------------------------------------

        // Check that a valid tag is given
        if ( branchName == null || StringUtils.isEmpty( branchName ) )
        {
            throw new ScmException( "Branch name must be specified." );
        }

        // Check if the whole working copy should be tagged
        if ( fileSet.getFileList().size() != 0 )
        {
            throw new ScmException( "This provider doesn't support branching subsets of a working copy." );
        }

        // Check that the working copy is in sync with the database
        MonotoneStatusCommand statusCmd = new MonotoneStatusCommand();
        statusCmd.setLogger( getLogger() );
        StatusScmResult statusResult = statusCmd.executeStatusCommand( repository, fileSet );

        if ( !statusResult.isSuccess() )
        {
            return new ScmResult( statusResult.getCommandLine(), "Error obtaining the status of the working copy.", statusResult.getCommandOutput(), false );
        }

        if ( statusResult.getChangedFiles().size() != 0 )
        {
            throw new ScmException( "This provider doesn't support branching of a working copy containing modified entries." );
        }

        //-----------------------------------------------------------------------------------------
        //  Get base revision
        //-----------------------------------------------------------------------------------------

        MonotoneBaseRevisionCommand baseRevisionCmd = new MonotoneBaseRevisionCommand();
        baseRevisionCmd.setLogger( getLogger() );
        BaseRevisionScmResult revisionResult = baseRevisionCmd.executeBaseRevisionCommand( repository, fileSet );

        if ( !revisionResult.isSuccess() )
        {
            return new ScmResult( revisionResult.getCommandLine(), "Error obtaining the base revision of the working copy.", revisionResult.getCommandOutput(), false );
        }

        //-----------------------------------------------------------------------------------------
        //  branch workspace
        //-----------------------------------------------------------------------------------------

        Command cmd = createCommand( repository, revisionResult.getBaseRevision(), branchName );

        try
        {
            Invoker invoker = new StandaloneInvoker( "mtn",fileSet.getBasedir().getAbsolutePath() );
            invoker.invoke( cmd );
        }
        catch ( InvokationException e )
        {
            if ( e.getType() == InvokationException.Type.PRECONDITION_ERROR )
            {
                return new BranchScmResult( "No commandline cvailable", "Precondition error during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.PROCESS_CREATION_ERROR )
            {
                return new BranchScmResult( e.getCommandline(), "Error while crreating the external process during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.EXECUTION_ERROR )
            {
                return new BranchScmResult( e.getCommandline(), "Error during command execution.", e.getFormatedErrorMessage(), false );
            }
        }

        return new BranchScmResult( MonotoneCommandlineUtils.getWorkingCopyFiles( fileSet.getBasedir() ), new ScmResult( null, null, null, true ) );
    }

    //-----------------------------------------------------------------------------------------
    //  MonotoneBranchCommand implementation
    //-----------------------------------------------------------------------------------------

    Command createCommand( ScmProviderRepository repository, String revision, String branch )
        throws ScmRepositoryException
    {
        MonotoneScmProviderRepository repo = ((MonotoneScmProviderRepository)repository);

        // Create base command
        CertCommand cmd = new CertCommand();

        // Set base options
        if ( repo.getConfigDirectory() != null && StringUtils.isNotEmpty( repo.getConfigDirectory() ) )
        {
            cmd.setConfigDirPath( repo.getConfigDirectory() );
        }

        if ( repo.getKeyringDirectory() != null && StringUtils.isNotEmpty( repo.getKeyringDirectory() ) )
        {
            cmd.setKeyDirPath( repo.getKeyringDirectory() );
        }

        if ( repo.getKey() != null && StringUtils.isNotEmpty( repo.getKey() ) )
        {
            cmd.setKey( repo.getKey() );
        }

        cmd.setRevision( revision );
        cmd.setCertificateName( "branch" );
        cmd.setCertificateValue( branch );

        return cmd;
    }
}
