package org.criticalsection.maven.scm.provider.monotone.command.add;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.command.add.AbstractAddCommand;
import org.apache.maven.scm.command.add.AddScmResult;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.ScmResult;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.ScmException;
import org.apache.maven.scm.ScmFileStatus;
import org.apache.maven.scm.ScmFile;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.codehaus.plexus.util.StringUtils;
import org.criticalsection.mtn4java.commands.Command;
import org.criticalsection.mtn4java.commands.invokers.Invoker;
import org.criticalsection.mtn4java.commands.invokers.StandaloneInvoker;
import org.criticalsection.mtn4java.commands.invokers.InvokationException;
import org.criticalsection.mtn4java.commands.standalone.AddCommand;

import java.io.File;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;

public class MonotoneAddCommand
    extends AbstractAddCommand
{

    //-----------------------------------------------------------------------------------------
    //  AbstractAddCommand implementation
    //-----------------------------------------------------------------------------------------

    protected ScmResult executeAddCommand( ScmProviderRepository repository, ScmFileSet fileSet, String message, boolean binary )
        throws ScmException
    {
        getLogger().debug( "Executing add command" );

        //-----------------------------------------------------------------------------------------
        //  Precondition checks
        //-----------------------------------------------------------------------------------------

        if ( binary )
        {
            throw new ScmException( "This provider does not support binary files" );
        }

        if ( fileSet.getFileList().size() == 0 )
        {
            throw new ScmException( "You must provide at least one file/directory to add" );
        }

        //-----------------------------------------------------------------------------------------
        //  add files to manifest
        //-----------------------------------------------------------------------------------------

        Command cmd = createCommand( repository, fileSet.getFileList() );

        try
        {
            Invoker invoker = new StandaloneInvoker( "mtn",fileSet.getBasedir().getAbsolutePath() );
            invoker.invoke( cmd );
        }
        catch ( InvokationException e )
        {
            if ( e.getType() == InvokationException.Type.PRECONDITION_ERROR )
            {
                return new AddScmResult( "No commandline cvailable", "Precondition error during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.PROCESS_CREATION_ERROR )
            {
                return new AddScmResult( e.getCommandline(), "Error while crreating the external process during command execution.", e.getFormatedErrorMessage(), false );
            }
            else if ( e.getType() == InvokationException.Type.EXECUTION_ERROR )
            {
                return new AddScmResult( e.getCommandline(), "Error during command execution.", e.getFormatedErrorMessage(), false );
            }
        }

        // Add filespaths
        List addedFiles = new ArrayList();
        List result = (List) cmd.getResult();

        for ( Iterator i = result.iterator(); i.hasNext(); )
        {
            String curFile = (String) i.next();
            addedFiles.add( new ScmFile( curFile, ScmFileStatus.ADDED ) );
        }

        return new AddScmResult( addedFiles, new ScmResult( null, null, null, true ) );
    }

    //-----------------------------------------------------------------------------------------
    //  MonotoneAddCommand implementation
    //-----------------------------------------------------------------------------------------

    Command createCommand( ScmProviderRepository repository, List files )
        throws ScmRepositoryException
    {
        MonotoneScmProviderRepository repo = ((MonotoneScmProviderRepository)repository);

        // Create base command
        AddCommand cmd = new AddCommand();

        // Set base options
        if ( repo.getConfigDirectory() != null && StringUtils.isNotEmpty( repo.getConfigDirectory() ) )
        {
            cmd.setConfigDirPath( repo.getConfigDirectory() );
        }

        if ( repo.getKeyringDirectory() != null && StringUtils.isNotEmpty( repo.getKeyringDirectory() ) )
        {
            cmd.setKeyDirPath( repo.getKeyringDirectory() );
        }

        if ( repo.getKey() != null && StringUtils.isNotEmpty( repo.getKey() ) )
        {
            cmd.setKey( repo.getKey() );
        }

        // Add filespaths
        List filePaths = new ArrayList();

        for ( Iterator i = files.iterator(); i.hasNext(); )
        {
            File curFile = (File) i.next();
            filePaths.add( curFile.getPath() );
        }

        cmd.setPathsToAdd( filePaths );

        return cmd;
    }
}
