package org.criticalsection.maven.scm.provider.monotone;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.provider.AbstractScmProvider;
import org.apache.maven.scm.provider.ScmProviderRepository;
import org.apache.maven.scm.repository.ScmRepositoryException;
import org.apache.maven.scm.repository.UnknownRepositoryStructure;
import org.apache.maven.scm.command.checkout.CheckOutScmResult;
import org.apache.maven.scm.command.add.AddScmResult;
import org.apache.maven.scm.command.status.StatusScmResult;
import org.apache.maven.scm.command.checkin.CheckInScmResult;
import org.apache.maven.scm.command.tag.TagScmResult;
import org.apache.maven.scm.command.branch.BranchScmResult;
import org.apache.maven.scm.ScmFileSet;
import org.apache.maven.scm.CommandParameters;
import org.apache.maven.scm.ScmException;
import org.criticalsection.maven.scm.provider.monotone.repository.MonotoneScmProviderRepository;
import org.criticalsection.maven.scm.provider.monotone.command.checkout.MonotoneCheckOutCommand;
import org.criticalsection.maven.scm.provider.monotone.command.add.MonotoneAddCommand;
import org.criticalsection.maven.scm.provider.monotone.command.status.MonotoneStatusCommand;
import org.criticalsection.maven.scm.provider.monotone.command.checkin.MonotoneCheckInCommand;
import org.criticalsection.maven.scm.provider.monotone.command.tag.MonotoneTagCommand;
import org.criticalsection.maven.scm.provider.monotone.command.branch.MonotoneBranchCommand;
import org.criticalsection.maven.scm.provider.monotone.MonotoneSettingsProvider;

import java.io.File;

/**
 * @plexus.component
 *   role="org.apache.maven.scm.provider.ScmProvider"
 *   role-hint="monotone"
 */
public class MonotoneScmProvider extends AbstractScmProvider
{
    /**
     * @plexus.requirement
     */
    private MonotoneSettingsProvider settingsProvider;

    public String getScmSpecificFilename()
    {
        this.getLogger().info( "getScmSpecificFilename()" );
        return "_MTN";
    }

    public String getScmType()
    {
        this.getLogger().info( "getScmType()" );
        return "monotone";
    }

    /**
     * @see org.apache.maven.scm.provider.AbstractScmProvider#makeProviderScmRepository(java.io.File)
     */
    public ScmProviderRepository makeProviderScmRepository( String scmSpecificUrl, char delimiter )
        throws ScmRepositoryException
    {
        return new MonotoneScmProviderRepository( scmSpecificUrl, settingsProvider );
    }

    /**
     * @see org.apache.maven.scm.provider.AbstractScmProvider#makeProviderScmRepository(java.io.File)
     */
    public ScmProviderRepository makeProviderScmRepository( File path )
        throws ScmRepositoryException, UnknownRepositoryStructure
    {
        if ( path == null || !path.isDirectory() )
        {
            throw new NullPointerException( "Path argument is null" );
        }

        if ( !path.isDirectory() )
        {
            throw new ScmRepositoryException( path.getAbsolutePath() + " isn't a valid directory." );
        }

        if ( !new File( path, getScmSpecificFilename() ).exists() )
        {
            throw new ScmRepositoryException( path.getAbsolutePath() + " isn't a monotone workspace directory." );
        }

        try
        {
            return makeProviderScmRepository( getRepositoryURL( path ), '|' );
        }
        catch ( ScmException e )
        {
            // XXX We should allow throwing of SCMException.
            throw new ScmRepositoryException( "Error executing info command", e );
        }
    }

    protected String getRepositoryURL( File path ) throws ScmException
    {
        return null;
    }

    /**
     * @see AbstractScmProvider#checkout(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    public CheckOutScmResult checkout( ScmProviderRepository repository, ScmFileSet fileSet,
                                       CommandParameters parameters )
        throws ScmException
    {
        MonotoneCheckOutCommand command = new MonotoneCheckOutCommand();

        command.setLogger( getLogger() );

        return (CheckOutScmResult) command.execute( repository, fileSet, parameters );
    }

    /**
     * @see AbstractScmProvider#add(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    public AddScmResult add( ScmProviderRepository repository, ScmFileSet fileSet, CommandParameters parameters )
        throws ScmException
    {
        MonotoneAddCommand command = new MonotoneAddCommand();

        command.setLogger( getLogger() );

        return (AddScmResult) command.execute( repository, fileSet, parameters );
    }

    /**
     * @see AbstractScmProvider#status(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    public StatusScmResult status( ScmProviderRepository repository, ScmFileSet fileSet, CommandParameters parameters )
        throws ScmException
    {
        MonotoneStatusCommand command = new MonotoneStatusCommand();

        command.setLogger( getLogger() );

        return (StatusScmResult) command.execute( repository, fileSet, parameters );
    }

    /**
     * @see AbstractScmProvider#checkin(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    public CheckInScmResult checkin( ScmProviderRepository repository, ScmFileSet fileSet, CommandParameters parameters )
        throws ScmException
    {
        MonotoneCheckInCommand command = new MonotoneCheckInCommand();

        command.setLogger( getLogger() );

        return (CheckInScmResult) command.execute( repository, fileSet, parameters );
    }

    /**
     * @see AbstractScmProvider#tag(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    public TagScmResult tag( ScmProviderRepository repository, ScmFileSet fileSet, CommandParameters parameters )
        throws ScmException
    {
        MonotoneTagCommand command = new MonotoneTagCommand();

        command.setLogger( getLogger() );

        return (TagScmResult) command.execute( repository, fileSet, parameters );
    }

    /**
     * @see AbstractScmProvider#branch(ScmProviderRepository,ScmFileSet,CommandParameters)
     */
    protected BranchScmResult branch( ScmProviderRepository repository, ScmFileSet fileSet,
                                      CommandParameters parameters )
        throws ScmException
    {
        MonotoneBranchCommand command = new MonotoneBranchCommand();

        command.setLogger( getLogger() );

        return (BranchScmResult) command.execute( repository, fileSet, parameters );
    }
}
