package org.criticalsection.maven.scm.provider.monotone;
/*
 * Copyright (C) 2008 Tim Kettler
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import org.apache.maven.scm.repository.ScmRepositoryException;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

import java.io.File;
import java.io.FileReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.FileInputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import org.criticalsection.maven.scm.providers.monotone.settings.Settings;
import org.criticalsection.maven.scm.providers.monotone.settings.io.xpp3.MtnXpp3Reader;


/**
 * The standard implementation of the settings provider for this scm provider.
 * <p>
 * This implementaion gathers the configuration options from different places with a defined priority.
 * <p>
 * The available places for specifying configuration options are:
 * <ul
 *   <li>Via system properties.</li>
 *   <li>In a file <i>monotone-settings.xml</i> located in the <i>~/.m2</i> directory</li>
 *   <li>gathered automatically from a _MTN/options file.</li>
 * </ul>
 *
 * If an option is specified in more than one place, the place with the highest priority takes
 * precedence. The places are ordered in descending priority in the above list.
 *
 * @author Tim Kettler <tik@criticalsection.org>
 *
 * @plexus.component
 *   role="org.criticalsection.maven.scm.provider.monotone.MonotoneSettingsProvider"
 */
public class DefaultMonotoneSettingsProvider implements MonotoneSettingsProvider
{
    private static final Pattern DATABASE_ENTRY = Pattern.compile( "^.*database \"(.*)\"$" );

    private static final Pattern KEY_ENTRY = Pattern.compile( "^.*key \"(.*)\"$" );

    private static final Pattern KEYDIR_ENTRY = Pattern.compile( "^.*keydir \"(.*)\"$" );

    private Settings settings;

    private String workspaceDatabase;

    private String workspaceKeydir;

    private String workspaceKey;

    private String configDirectory;

    private String keyringDirectory;

    private String database;

    private String key;

    public DefaultMonotoneSettingsProvider()
    {
        this.settings = getSettings();
        getWorkspaceOptions();
    }

    public String getConfigDirectory()
    {
        if ( configDirectory == null && System.getProperty( CONFIG_DIR_PROPERTY ) != null )
        {
            configDirectory = System.getProperty( CONFIG_DIR_PROPERTY );
        }

        if ( configDirectory == null && settings != null && settings.getConfigDirectory() != null )
        {
            configDirectory = settings.getConfigDirectory();
        }

        return configDirectory;
    }

    public String getKeyringDirectory()
    {
        if ( keyringDirectory == null && System.getProperty( KEYRING_DIR_PROPERTY ) != null )
        {
            keyringDirectory = System.getProperty( KEYRING_DIR_PROPERTY );
        }

        if ( keyringDirectory == null && settings != null && settings.getKeyringDirectory() != null )
        {
            keyringDirectory = settings.getConfigDirectory();
        }

        if ( keyringDirectory == null && workspaceKeydir != null )
        {
            keyringDirectory = workspaceKeydir;
        }

        return keyringDirectory;
    }

    public String getDatabase() throws ScmRepositoryException
    {
        if ( database == null && System.getProperty( DATABASE_PROPERTY ) != null )
        {
            database = System.getProperty( DATABASE_PROPERTY );
        }

        if ( database == null && settings != null && settings.getDatabase() != null )
        {
            database = settings.getDatabase();
        }

        if ( database == null && workspaceDatabase != null )
        {
            database = workspaceDatabase;
        }

        return database;
    }


    public String getKey() throws ScmRepositoryException
    {
        if ( key == null && System.getProperty( KEY_PROPERTY ) != null )
        {
            key = System.getProperty( KEY_PROPERTY );
        }

        if ( key == null && settings != null && settings.getKey() != null )
        {
            key = settings.getKey();
        }

        if ( key == null && workspaceKey != null )
        {
            key = workspaceKey;
        }

        if ( key == null )
        {
            throw new ScmRepositoryException( "No key specified." );
        }

        return key;
    }

    private Settings getSettings()
    {
        File scmUserDir = new File( System.getProperty( "user.home" ), ".scm" );
        File settingsFile = new File( scmUserDir, "mtn-settings.xml" );

        if ( settingsFile.exists() )
        {
            MtnXpp3Reader reader = new MtnXpp3Reader();
            try
            {
                return reader.read( new FileReader( settingsFile ) );
            }
            catch ( FileNotFoundException e )
            {
                // nothing to do
            }
            catch ( IOException e )
            {
                // nothing to do
            }
            catch ( XmlPullParserException e )
            {
                StringBuffer message = new StringBuffer();
                message.append( settingsFile.getAbsolutePath() );
                message.append( " isn't well formed. SKIPPED:\n  " );
                message.append( e.getMessage() );

                System.out.println( message.toString() );
            }
        }

        return new Settings();
    }

    private void getWorkspaceOptions()
    {
        String workspaceRoot = MonotoneCommandlineUtils.searchWorkspaceRoot( "." );

        if (  workspaceRoot != null )
        {
            File options = new File( workspaceRoot + File.separator + "_MTN" + File.separator + "options" );

            BufferedReader reader;

            try
            {
                reader = new BufferedReader( new InputStreamReader( new FileInputStream( options ) ) );
            }
            catch ( FileNotFoundException e )
            {
                // skip the parsing if the file doesn't exist
                return;
            }

            String line;

            try
            {
                while( ( line = reader.readLine() ) != null )
                {
                    if ( DATABASE_ENTRY.matcher( line ).matches() )
                    {
                        Matcher m = DATABASE_ENTRY.matcher( line );
                        m.matches();
                        workspaceDatabase = m.group( 1 );
                    }
                    if ( KEY_ENTRY.matcher( line ).matches() )
                    {
                        Matcher m = KEY_ENTRY.matcher( line );
                        m.matches();
                        workspaceKey = m.group( 1 );
                    }
                    if ( KEYDIR_ENTRY.matcher( line ).matches() )
                    {
                        Matcher m = KEYDIR_ENTRY.matcher( line );
                        m.matches();
                        workspaceKeydir = m.group( 1 );
                    }
                }
            }
            catch ( IOException e )
            {
                // nothing to do here
            }

            try
            {
                reader.close();
            }
            catch ( IOException e )
            {
                // nothing to do here
            }
        }
    }
}
