/*
 * Copyright (C) 2007-2019 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.validator.impl;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.text.StringSubstitutor;
import org.craftercms.commons.entitlements.exception.EntitlementExceededException;
import org.craftercms.commons.entitlements.exception.EntitlementException;
import org.craftercms.commons.entitlements.exception.ExpiredLicenseException;
import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.exception.UnsupportedEntitlementException;
import org.craftercms.commons.entitlements.manager.LicenseAware;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.model.Entitlement;
import org.craftercms.commons.entitlements.model.EntitlementType;
import org.craftercms.commons.entitlements.model.License;
import org.craftercms.commons.entitlements.model.LicenseType;
import org.craftercms.commons.entitlements.model.Module;
import org.craftercms.commons.entitlements.usage.EntitlementUsageProvider;
import org.craftercms.commons.entitlements.validator.EntitlementValidator;
import org.slf4j.Logger;

import static java.time.temporal.ChronoUnit.DAYS;

/**
 * Implementation of {@link EntitlementValidator} for Crafter Enterprise Edition.
 *
 * @author joseross
 */
public class DefaultEntitlementValidatorImpl extends AbstractLicenseLoader
    implements EntitlementValidator, LicenseAware {

    private static final Logger logger = LicenseManager.getLogger();

    /**
     * Object loaded from the license file.
     */
    protected License license;

    /**
     * {@link EntitlementUsageProvider} being used by the current module.
     */
    protected EntitlementUsageProvider provider;

    public void setProvider(final EntitlementUsageProvider provider) {
        this.provider = provider;
    }

    /**
     * Performs an early validation of the license file.
     * @throws EntitlementException if the license file is not found or can't be read.
     */
    public void init() throws EntitlementException {
        logger.info("Crafter CMS v{} Enterprise Edition", getPackageVersion());
        loadLicense();
    }

    public void loadLicense() throws InvalidLicenseException {
        logger.info("Loading license");
        try {
            license = decryptLicense();
            if(license != null) {
                DateTimeFormatter formatter = DateTimeFormatter.ISO_LOCAL_DATE.withZone(ZoneId.of("UTC"));
                logger.info("Product licensed to: {}", license.getClientName());
                logger.info("License valid from {} through {}", formatter.format(license.getContractStartDate()),
                    formatter.format(license.getContractEndDate()));
                logger.info("License for {} environments", license.getEnvironment());
                Map<Module, List<Entitlement>> modules = license.getEntitlements();
                if(modules != null && !modules.isEmpty()) {
                    logger.info("License entitlements:");
                    modules.forEach((module, entitlements) -> {
                        if(entitlements == null || entitlements.isEmpty()) {
                            return;
                        }
                        logger.info("{}:", module);
                        entitlements.forEach(entitlement -> {
                            if (entitlement.getValue() < 0) {
                                logger.info("  {}: UNLIMITED", entitlement.getType());
                            } else if (entitlement.getValue() > 0) {
                                logger.info("  {}: {}", entitlement.getType(), entitlement.getValue());
                            }
                        });
                    });
                }
                try {
                    validateLicense();
                } catch (ExpiredLicenseException e) {
                    logger.warn("License has expired, some features may be disabled.", e);
                }
            }
        } catch (Exception e) {
            logger.error("License found but could not be loaded, unable to start", e);
            throw new InvalidLicenseException("License found but could not be loaded", e);
        }
    }

    /**
     * Performs general validations on the license object.
     * @throws EntitlementException if any validations fails
     */
    public void validateLicense() throws EntitlementException {
        if(license.getLicenseType() == LicenseType.STANDARD &&
            license.getContractEndDate().truncatedTo(DAYS).isBefore(Instant.now().truncatedTo(DAYS))) {
            throw new ExpiredLicenseException(license.getContractEndDate());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void validateEntitlement(EntitlementType entitlementType, int newAmount)
        throws EntitlementException {
        Module module = provider.getModule();
        if (!provider.getSupportedEntitlements().contains(entitlementType)) {
            logger.error("Unsupported entitlement {} for module {}", entitlementType, module);
            throw new UnsupportedEntitlementException(module, entitlementType);
        }
        int currentValue = provider.getEntitlementUsage(entitlementType);
        logger.debug("Validating entitlement {} for module {} with value {}", entitlementType, module, currentValue);
        try {
            validateLicense();
        } catch (ExpiredLicenseException e) {
            logger.warn("The current license has already expired");
            // Engine should be allowed to display sites even if the license is expired.
            if(license.isReadOnlyEnabled() && module != Module.ENGINE) {
                logger.error("The system is in read-only mode", e);
                throw e;
            }
        }
        List<Entitlement> entitlements = license.getEntitlements().get(module);
        if(entitlements == null) {
            throw new UnsupportedEntitlementException(module, entitlementType);
        }

        Entitlement entitlement = entitlements
                                            .stream()
                                            .filter(e -> e.getType().equals(entitlementType))
                                            .findFirst()
                                            .orElseThrow(() ->
                                                new UnsupportedEntitlementException(module, entitlementType));

        int entitlementValue = entitlement.getValue();

        if(entitlementValue != -1 && currentValue + newAmount > entitlementValue) {
            logger.error("Exceeded entitlement '{}' for module '{}': using '{}' of '{}'",
                entitlement, module, currentValue, entitlementValue);
            throw new EntitlementExceededException(module, entitlementType, entitlementValue, currentValue);
        }
        logger.debug("Validated entitlement '{}' for module '{}': using '{}' of '{}'",
            entitlement, module, currentValue, entitlementValue);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getId() {
        return license.getId();
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public long getClientId() {
        return license.getClientId();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getVersion() {
        return license.getVersion();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDescription() {
        Instant endDate = license.getContractEndDate().truncatedTo(DAYS);
        Map<String, Object> values = new HashMap<>();
        values.put("version", getPackageVersion());
        Instant now = Instant.now().truncatedTo(DAYS);
        String encoded;
        if(endDate.isAfter(now.plus(30, DAYS))) {
            encoded = "UG93ZXJlZCBieSBDcmFmdGVyIENNUyBFbnRlcnByaXNlIEVkaXRpb24gdiR7dmVyc2lvbn0uIFN1cHBvc"
                + "nRlZCB0aHJvdWdoICR7ZGF0ZX0uIDxhIGhyZWY9Imh0dHBzOi8vY3JhZnRlcnNvZnR3YXJlLnplbmRlc2"
                + "suY29tIiB0YXJnZXQ9Il9ibGFuayI+R2V0IHN1cHBvcnQ8L2E+LiA8YSBocmVmPSJodHRwczovL2NyYWZ"
                + "0ZXJzb2Z0d2FyZS5jb20vYmxvZyIgdGFyZ2V0PSJfYmxhbmsiPkNyYWZ0ZXIgTmV3czwvYT4u";
            values.put("date", DateTimeFormatter.ISO_LOCAL_DATE.withZone(ZoneId.of("UTC")).format(endDate));
        } else if(endDate.isAfter(now)) {
            encoded = "UG93ZXJlZCBieSBDcmFmdGVyIENNUyBFbnRlcnByaXNlIEVkaXRpb24gdiR7dmVyc2lvbn0uIFN1cHBvc"
                + "nQgZXhwaXJlcyBpbiAke2RheXN9IGRheXMsIDxhIGhyZWY9Imh0dHBzOi8vY3JhZnRlcnNvZnR3YXJlLn"
                + "plbmRlc2suY29tIiB0YXJnZXQ9Il9ibGFuayI+Y29udGFjdCBzdXBwb3J0IHRvIHJlbmV3PC9hPiEgPGEgaH"
                + "JlZj0iaHR0cHM6Ly9jcmFmdGVyc29mdHdhcmUuY29tL2Jsb2ciIHRhcmdldD0iX2JsYW5rIj5DcmFmdGVyI"
                + "E5ld3M8L2E+Lg==";
            values.put("days", Duration.between(now, endDate).toDays());
        } else {
            encoded = "UG93ZXJlZCBieSBDcmFmdGVyIENNUyBFbnRlcnByaXNlIEVkaXRpb24gdiR7dmVyc2lvbn0uIFVuc3Vwc"
                + "G9ydGVkISA8YSBocmVmPSJtYWlsdG86c2FsZXNAY3JhZnRlcnNvZnR3YXJlLmNvbSIgdGFyZ2V0PSJf"
                + "YmxhbmsiPkNvbnRhY3QgdXMgdG8gZ2V0IHN1cHBvcnQ8L2E+LiA8YSBocmVmPSJodHRwczovL2NyY"
                + "WZ0ZXJzb2Z0d2FyZS5jb20vYmxvZyIgdGFyZ2V0PSJfYmxhbmsiPkNyYWZ0ZXIgTmV3czwvYT4u";
        }
        StringSubstitutor stringSubstitutor = new StringSubstitutor(values);
        return stringSubstitutor.replace(new String(Base64.getDecoder().decode(encoded)));
    }

}