/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.usage.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.craftercms.commons.crypto.PGPUtils;
import org.craftercms.commons.entitlements.exception.EntitlementException;
import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.manager.LicenseAware;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.model.License;
import org.craftercms.commons.entitlements.usage.EntitlementUsage;
import org.craftercms.commons.entitlements.usage.EntitlementUsageProvider;
import org.craftercms.commons.entitlements.usage.EntitlementUsageSender;
import org.craftercms.commons.entitlements.validator.EntitlementValidator;
import org.slf4j.Logger;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import com.sun.management.OperatingSystemMXBean;

/**
 * Default implementation for {@link EntitlementUsageSender} for Crafter Enterprise Edition.
 *
 * @author joseross
 */
public class DefaultEntitlementUsageSenderImpl extends AbstractLicenseLoader
    implements EntitlementUsageSender, LicenseAware, DisposableBean {

    private static final Logger logger = LicenseManager.getLogger();

    /**
     * Content of the public publicKeyContent file.
     */
    protected byte[] publicKeyContent;

    protected License license;

    /**
     * UPD Socket used to send the data.
     */
    protected DatagramSocket socket;

    /**
     * Name of the host id file.
     */
    protected String hostIdName;

    /**
     * Value of the host id.
     */
    protected String hostId;

    /**
     * {@link EntitlementValidator} being used by the current module.
     */
    protected EntitlementValidator entitlementValidator;

    /**
     * {@link EntitlementUsageProvider} being used by the current module.
     */
    protected EntitlementUsageProvider provider;

    /**
     * {@link TaskScheduler} being used by the current module.
     */
    protected TaskScheduler taskScheduler;

    /**
     * {@link ScheduledFuture} instance to keep track of the scheduled task.
     */
    protected ScheduledFuture<?> scheduledTask;

    @Required
    public void setHostIdName(final String hostIdName) {
        this.hostIdName = hostIdName;
    }

    @Required
    public void setEntitlementValidator(final EntitlementValidator entitlementValidator) {
        this.entitlementValidator = entitlementValidator;
    }

    @Required
    public void setProvider(final EntitlementUsageProvider provider) {
        this.provider = provider;
    }

    @Required
    public void setTaskScheduler(final TaskScheduler taskScheduler) {
        this.taskScheduler = taskScheduler;
    }

    /**
     * Performs an early validation of the PGP Public Key.
     * @throws SocketException if there is an error opening the socket
     * @throws EntitlementException if there is an error reading or decrypting the license file
     */
    public void init() throws IOException, EntitlementException {
        socket = new DatagramSocket();
        loadLicense();
        loadHostId();
    }

    /**
     * Send the usage data when the app context is ready
     */
    @EventListener
    public void onContextReady(ContextRefreshedEvent event) {
        sendData();
    }

    /**
     * Loads the host id or creates a new one if needed.
     * @throws IOException if there is an error reading or writing to the host id file.
     */
    protected void loadHostId() throws IOException {
        Resource idFile = licenseFile.createRelative(hostIdName);
        if(idFile.exists()) {
            logger.debug("Using existing host id");
            try (InputStream is = idFile.getInputStream()) {
                hostId = IOUtils.toString(is, Charset.defaultCharset());
                if(StringUtils.isEmpty(hostId)) {
                    throw new IllegalStateException("Empty host id file");
                }
            } catch (Exception e) {
                logger.error("Could not read host id, will generate a new one");
            }
        } else {
            logger.debug("Generating new host id");
            hostId = UUID.randomUUID().toString();

            try {
                Files.write(
                    Paths.get(licenseFile.getURI()).toAbsolutePath().getParent().resolve(hostIdName),
                    hostId.getBytes()
                );
            } catch (Exception e) {
                logger.error("Could not write host id, using a temporary value");
            }
        }

        logger.info("Host ID: {}", hostId);
    }

    public void loadLicense() throws InvalidLicenseException {
        logger.info("Loading license");
        try {
            license = decryptLicense();
            publicKeyContent = readPublicKey();
            configureScheduledTask();
        } catch (Exception e) {
            logger.error("License found but could not be loaded, unable to start", e);
            throw new InvalidLicenseException("License found but could not be loaded", e);
        }
    }

    public void configureScheduledTask() {
        logger.debug("Configuring scheduled task");
        if(scheduledTask != null) {
            logger.debug("Canceling existing scheduled task");
            scheduledTask.cancel(false);
        }
        String cronExpression = license.getConfiguration().getCron();
        logger.debug("Adding new scheduled task with cron expression: {}", cronExpression);
        scheduledTask = taskScheduler.schedule(this::sendData, new CronTrigger(cronExpression));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public EntitlementUsage prepareUsageData() {
        EntitlementUsage usage = new EntitlementUsage();

        try(InputStream is = Files.newInputStream(licenseFile.getFile().toPath())) {
            usage.setAdditionalData(DigestUtils.md5Hex(is).getBytes());
        } catch (IOException e) {
            logger.warn("Could not calculate md5 for license file", e);
        }

        usage.setValidatorId(entitlementValidator.getId());
        usage.setClientId(entitlementValidator.getClientId());
        usage.setHostId(hostId);
        usage.setValidatorVersion(entitlementValidator.getVersion());
        usage.setModule(provider.getModule());
        usage.setModuleVersion(entitlementValidator.getPackageVersion());
        usage.setModuleBuild(entitlementValidator.getPackageBuild());
        usage.setEntitlements(provider.getCurrentUsage());

        usage.setNetworkInterfaces(getNetworkInterfaces());

        try {
            RuntimeMXBean runtime = ManagementFactory.getRuntimeMXBean();
            long startTime = runtime.getStartTime();
            long upTime = runtime.getUptime();

            usage.setCores(Runtime.getRuntime().availableProcessors());
            usage.setTotalJvmMemory(Runtime.getRuntime().totalMemory());
            usage.setFreeJvmMemory(Runtime.getRuntime().freeMemory());

            OperatingSystemMXBean os = (com.sun.management.OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();


            usage.setTotalPhysicalMemory(os.getTotalPhysicalMemorySize());
            usage.setFreePhysicalMemory(os.getFreePhysicalMemorySize());
            usage.setTotalSwapMemory(os.getTotalSwapSpaceSize());
            usage.setFreeSwapMemory(os.getFreeSwapSpaceSize());

            usage.setStartupDate(Instant.ofEpochMilli(startTime));
            usage.setRunDuration(upTime);
            usage.setOsName(os.getName());
            usage.setOsVersion(os.getVersion());
            usage.setOsArch(os.getArch());
        } catch (Exception e) {
            logger.warn("Unknown error collecting host information", e);
        }

        return usage;
    }

    /**
     * Collects the information of all network interfaces available.
     * @return list of network interfaces
     */
    protected List<EntitlementUsage.NetworkInterface> getNetworkInterfaces() {
        List<EntitlementUsage.NetworkInterface> result = new LinkedList<>();
        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            while (interfaces.hasMoreElements()) {
                NetworkInterface ni = interfaces.nextElement();
                logger.debug("Checking network interface {}", ni.getName());
                Enumeration<InetAddress> addresses = ni.getInetAddresses();
                if(!ni.isLoopback() && addresses.hasMoreElements()) {
                    logger.debug("Network interface {} will be sent", ni.getName());
                    EntitlementUsage.NetworkInterface iface = new EntitlementUsage.NetworkInterface();
                    iface.setName(ni.getName());
                    InetAddress address = addresses.nextElement();
                    if (StringUtils.isNotEmpty(address.getHostAddress())) {
                        iface.setIpAddress(address.getHostAddress());
                    }
                    if (StringUtils.isNotEmpty(address.getHostName())) {
                        iface.setHostname(address.getHostName());
                    }
                    byte[] macAddress = ni.getHardwareAddress();
                    if (macAddress != null) {
                        StringBuilder sb = new StringBuilder();
                        for (byte b : macAddress) {
                            sb.append(String.format("%02x:", b));
                        }
                        sb.deleteCharAt(sb.length() - 1);
                        iface.setHardwareAddress(sb.toString());
                    }
                    result.add(iface);
                }
            }
        } catch (Exception e) {
            logger.warn("Could not get network interfaces information", e);
        }
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void sendData() {
        if(!license.getConfiguration().isEnabled()) {
            return;
        }
        try {
            EntitlementUsage usage = prepareUsageData();
            Path tempUsage = Files.createTempFile("crafter", "lic-usage");

            ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
            try(OutputStream out = Files.newOutputStream(tempUsage)) {
                objectMapper.writeValue(out, usage);
                PGPUtils.encrypt(tempUsage, new ByteArrayInputStream(publicKeyContent), encrypted);
                byte[] content = encrypted.toByteArray();
                DatagramPacket packet = new DatagramPacket(content, content.length,
                    InetAddress.getByName(
                        license.getConfiguration().getDomain()), license.getConfiguration().getPort());
                socket.send(packet);
            }
        } catch (IOException e) {
            logger.error("Error sending usage data", e);
        } catch (Exception e) {
            logger.error("Error collecting usage data", e);
        }
    }

    @Override
    public void destroy() {
        sendData();
    }

}
