/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.model;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.craftercms.commons.entitlements.usage.UsageSenderConfiguration;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;

/**
 * Holds the information read from the license file.
 *
 * @author joseross
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class License {

    /**
     * Current version used for the license file.
     */
    public static final String CURRENT_VERSION = "1.0";

    /**
     * Id of the license.
     */
    protected long id;

    /**
     * Version of the license file. Defaults to the current version.
     */
    protected String version = CURRENT_VERSION;

    /**
     * General label to describe the license.
     */
    protected String label;

    /**
     * Environment for the license.
     */
    protected Environment environment;

    /**
     * General notes to describe the license.
     */
    protected String notes;

    /**
     * Order Form Id of the license.
     */
    protected String orderFormId;

    /**
     * Name of the client.
     */
    protected String clientName;

    /**
     * Id of the client.
     */
    protected long clientId;

    /**
     * Start date of the license.
     */
    protected Instant contractStartDate;

    /**
     * End date of the license.
     */
    protected Instant contractEndDate;

    /**
     * Type of the license.
     */
    protected LicenseType licenseType;

    /**
     * Indicates if the system should be read-only when the license expires.
     */
    protected boolean readOnlyEnabled;

    /**
     * The username of the user that created the license.
     */
    protected String createdBy;

    /**
     * The date when the license was created.
     */
    protected Instant createdDate;

    /**
     * Map of entitlements for each module.
     */
    protected Map<Module, List<Entitlement>> entitlements;

    protected UsageSenderConfiguration configuration;

    public long getId() {
        return id;
    }

    public void setId(final long id) {
        this.id = id;
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(final String version) {
        this.version = version;
    }

    public String getLabel() {
        return label;
    }

    public void setLabel(final String label) {
        this.label = label;
    }

    public Environment getEnvironment() {
        return environment;
    }

    public void setEnvironment(final Environment environment) {
        this.environment = environment;
    }

    public String getNotes() {
        return notes;
    }

    public void setNotes(final String notes) {
        this.notes = notes;
    }

    public String getOrderFormId() {
        return orderFormId;
    }

    public void setOrderFormId(final String orderFormId) {
        this.orderFormId = orderFormId;
    }

    public String getClientName() {
        return clientName;
    }

    public void setClientName(String clientName) {
        this.clientName = clientName;
    }

    public long getClientId() {
        return clientId;
    }

    public void setClientId(long clientId) {
        this.clientId = clientId;
    }

    public Instant getContractStartDate() {
        return contractStartDate;
    }

    public void setContractStartDate(Instant contractStartDate) {
        this.contractStartDate = contractStartDate;
    }

    public Instant getContractEndDate() {
        return contractEndDate;
    }

    public void setContractEndDate(Instant contractEndDate) {
        this.contractEndDate = contractEndDate;
    }

    public LicenseType getLicenseType() {
        return licenseType;
    }

    public void setLicenseType(LicenseType licenseType) {
        this.licenseType = licenseType;
    }

    public boolean isReadOnlyEnabled() {
        return readOnlyEnabled;
    }

    public void setReadOnlyEnabled(final boolean readOnlyEnabled) {
        this.readOnlyEnabled = readOnlyEnabled;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public void setCreatedBy(final String createdBy) {
        this.createdBy = createdBy;
    }

    public Instant getCreatedDate() {
        return createdDate;
    }

    public void setCreatedDate(final Instant createdDate) {
        this.createdDate = createdDate;
    }

    public Map<Module, List<Entitlement>> getEntitlements() {
        return entitlements;
    }

    public void setEntitlements(final Map<Module, List<Entitlement>> entitlements) {
        this.entitlements = entitlements;
    }

    public UsageSenderConfiguration getConfiguration() {
        return configuration;
    }

    public void setConfiguration(final UsageSenderConfiguration configuration) {
        this.configuration = configuration;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof License)) {
            return false;
        }
        final License license = (License)o;
        return id == license.id && clientId == license.clientId && readOnlyEnabled == license.readOnlyEnabled &&
            Objects.equals(version, license.version) && Objects.equals(label, license.label) &&
            environment == license.environment && Objects.equals(notes, license.notes) &&
            Objects.equals(orderFormId, license.orderFormId) && Objects.equals(clientName, license.clientName) &&
            Objects.equals(contractStartDate, license.contractStartDate) &&
            Objects.equals(contractEndDate, license.contractEndDate) && licenseType == license.licenseType &&
            Objects.equals(createdBy, license.createdBy) && Objects.equals(createdDate, license.createdDate) &&
            Objects.equals(entitlements, license.entitlements) && Objects.equals(configuration, license.configuration);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, version, label, environment, notes, orderFormId, clientName, clientId,
            contractStartDate, contractEndDate, licenseType, readOnlyEnabled, createdBy, createdDate, entitlements,
            configuration);
    }

    @Override
    public String toString() {
        return "License{" + "id=" + id + ", version='" + version + '\'' + ", label='" + label + '\'' + ", environment"
            + "=" + environment + ", notes='" + notes + '\'' + ", orderFormId='" + orderFormId + '\'' + ", clientName"
            + "='" + clientName + '\'' + ", clientId=" + clientId + ", contractStartDate=" + contractStartDate + ", "
            + "contractEndDate=" + contractEndDate + ", licenseType=" + licenseType + ", readOnlyEnabled=" +
            readOnlyEnabled + ", createdBy='" + createdBy + '\'' + ", createdDate=" + createdDate + ", entitlements"
            + "=" + entitlements + ", configuration=" + configuration + '}';
    }
}
