/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.manager;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.io.IOUtils;
import org.craftercms.commons.crypto.PGPUtils;
import org.craftercms.commons.entitlements.model.License;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Provides basic operations to read license files.
 *
 * @author joseross
 */
public abstract class AbstractLicenseLoader {

    /**
     * The location of the license file to check.
     */
    protected Resource licenseFile;
    /**
     * Name of the license file.
     */
    protected String licenseName;
    /**
     * Name of the private publicKeyContent file.
     */
    protected String privateKeyName;
    /**
     * Passphrase for the PGP Private Key.
     */
    protected char[] privateKeyPassword;
    /**
     * Jackson {@link ObjectMapper} used to read from the license file.
     */
    protected ObjectMapper objectMapper;
    /**
     * Name of the public publicKeyContent file.
     */
    protected String publicKeyName;

    public AbstractLicenseLoader() {
        objectMapper = new ObjectMapper();
        objectMapper.findAndRegisterModules();
    }

    @Required
    public void setLicenseFile(final Resource licenseFile) {
        this.licenseFile = licenseFile;
    }

    @Required
    public void setLicenseName(final String licenseName) {
        this.licenseName = licenseName;
    }

    @Required
    public void setPrivateKeyName(final String privateKeyName) {
        this.privateKeyName = privateKeyName;
    }

    @Required
    public void setPrivateKeyPassword(final char[] privateKeyPassword) {
        this.privateKeyPassword = privateKeyPassword;
    }

    @Required
    public void setPublicKeyName(final String publicKeyName) {
        this.publicKeyName = publicKeyName;
    }

    /**
     * Reads the license file and decrypts the content.
     * @return the license object
     * @throws Exception if the license file can't be read or decrypted
     */
    protected License decryptLicense() throws Exception {
        try(ZipFile zip = new ZipFile(licenseFile.getFile())) {
            ZipEntry keyEntry = zip.getEntry(privateKeyName);
            ZipEntry licenseEntry = zip.getEntry(licenseName);
            ByteArrayOutputStream decrypted = new ByteArrayOutputStream();
            try(InputStream licenseStream = zip.getInputStream(licenseEntry);
                InputStream keyStream = zip.getInputStream(keyEntry)) {
                PGPUtils.decrypt(licenseStream, decrypted, keyStream, privateKeyPassword);
                return objectMapper.readValue(decrypted.toByteArray(), License.class);
            }
        }
    }

    /**
     * Reads the license file and returns the content of the public key.
     * @return content of the public key
     * @throws IOException if the file can't be read
     */
    protected byte[] readPublicKey() throws IOException {
        byte[] content;
        try (ZipFile zip = new ZipFile(licenseFile.getFile())) {
            ZipEntry keyEntry = zip.getEntry(publicKeyName);
            try (InputStream keyStream = zip.getInputStream(keyEntry)) {
                content = IOUtils.toByteArray(keyStream);
            }
        }
        return content;
    }

}
