/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.usage.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Base64;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.craftercms.commons.crypto.CryptoException;
import org.craftercms.commons.crypto.PGPUtils;
import org.craftercms.commons.crypto.TextEncryptor;
import org.craftercms.commons.crypto.impl.PbkAesTextEncryptor;
import org.craftercms.commons.entitlements.exception.EntitlementException;
import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.manager.LicenseAware;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.model.License;
import org.craftercms.commons.entitlements.usage.EntitlementUsage;
import org.craftercms.commons.entitlements.usage.EntitlementUsageProvider;
import org.craftercms.commons.entitlements.usage.EntitlementUsageSender;
import org.craftercms.commons.entitlements.validator.EntitlementValidator;
import org.slf4j.Logger;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import com.sun.management.OperatingSystemMXBean;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;


/**
 * Default implementation for {@link EntitlementUsageSender} for Crafter Enterprise Edition.
 *
 * @author joseross
 */
public class DefaultEntitlementUsageSenderImpl extends AbstractLicenseLoader
    implements EntitlementUsageSender, LicenseAware, DisposableBean {

    private static final Logger logger = LicenseManager.getLogger();

    /**
     * Key used to encrypt/decrypt the host id value
     */
    private static final String HOST_ID_RESOLVER = "XTMhNyZ9e1NBOW4kP3cqTQ==";

    /**
     * Salt used to encrypt/decrypt the host id value
     */
    private static final String HOST_ID_GENERATOR = "Tlg1MmF5OUhpNA==";

    /**
     * Content of the public publicKeyContent file.
     */
    protected byte[] publicKeyContent;

    protected License license;

    /**
     * UPD Socket used to send the data.
     */
    protected DatagramSocket socket;

    /**
     * Name of the host id file.
     */
    protected String hostIdName;

    /**
     * Location for the host id file
     */
    protected Path dataFolder;

    /**
     * Value of the host id.
     */
    protected String hostId;

    /**
     * {@link EntitlementValidator} being used by the current module.
     */
    protected EntitlementValidator entitlementValidator;

    /**
     * {@link EntitlementUsageProvider} being used by the current module.
     */
    protected EntitlementUsageProvider provider;

    /**
     * {@link TaskScheduler} being used by the current module.
     */
    protected TaskScheduler taskScheduler;

    /**
     * {@link ScheduledFuture} instance to keep track of the scheduled task.
     */
    protected ScheduledFuture<?> scheduledTask;

    private TextEncryptor encryptor;

    public DefaultEntitlementUsageSenderImpl(final String hostIdName, final Path dataFolder,
                                             final EntitlementValidator entitlementValidator,
                                             final EntitlementUsageProvider provider,
                                             final TaskScheduler taskScheduler) throws CryptoException {
        this.hostIdName = hostIdName;
        this.dataFolder = dataFolder;
        this.entitlementValidator = entitlementValidator;
        this.provider = provider;
        this.taskScheduler = taskScheduler;
        Base64.Decoder decoder = Base64.getDecoder();
        encryptor = new PbkAesTextEncryptor(new String(decoder.decode(HOST_ID_RESOLVER)),
                                            new String(decoder.decode(HOST_ID_GENERATOR)));
    }

    /**
     * Performs an early validation of the PGP Public Key.
     * @throws SocketException if there is an error opening the socket
     * @throws EntitlementException if there is an error reading or decrypting the license file
     */
    public void init() throws IOException, EntitlementException {
        socket = new DatagramSocket();
        loadLicense();
        loadHostId();
    }

    /**
     * Send the usage data when the app context is ready
     */
    @EventListener
    public void onContextReady(ContextRefreshedEvent event) {
        sendData();
    }

    /**
     * Loads the host id or creates a new one if needed.
     * @throws IOException if there is an error reading or writing to the host id file.
     */
    protected void loadHostId() throws IOException {
        boolean createFile = false;
        // check if there previous file exists
        Resource idFile = licenseFile.createRelative(hostIdName);
        if (idFile.exists()) {
            logger.debug("Migrating previous host id");
            // load the value
            try (InputStream is = idFile.getInputStream()) {
                hostId = IOUtils.toString(is, Charset.defaultCharset());
                createFile = true;
                idFile.getFile().delete();
            } catch (Exception e) {
                logger.debug("Error reading previous host id, a new one will be generated");
            }
        }

        // check the new location, if the previous existed it will be ignored to avoid issues
        Path hostFile = dataFolder.resolve(hostIdName);
        if (Files.exists(hostFile)) {
            // decrypt value & set it
            try {
                hostId = encryptor.decrypt(new String(Files.readAllBytes(hostFile)));
                createFile = false;
            } catch (Exception e) {
                logger.info("Could not read host id, will generate a new one");
            }
        }

        if (isEmpty(hostId)) {
            // try to create it
            logger.debug("Generating new host id");
            hostId = UUID.randomUUID().toString();
            createFile = true;
        }

        if (createFile) {
            // save the value
            try {
                Files.write(dataFolder.resolve(hostIdName), encryptor.encrypt(hostId).getBytes());
            } catch (Exception e) {
                logger.info("Could not write host id, using a temporary value");
            }
        }

        logger.info("Host ID: {}", hostId);
    }

    public void loadLicense() throws InvalidLicenseException {
        logger.info("Loading license");
        try {
            license = decryptLicense();
            publicKeyContent = readPublicKey();
            configureScheduledTask();
        } catch (Exception e) {
            logger.error("License found but could not be loaded, unable to start", e);
            throw new InvalidLicenseException("License found but could not be loaded", e);
        }
    }

    public void configureScheduledTask() {
        logger.debug("Configuring scheduled task");
        if(scheduledTask != null) {
            logger.debug("Canceling existing scheduled task");
            scheduledTask.cancel(false);
        }
        String cronExpression = license.getConfiguration().getCron();
        logger.debug("Adding new scheduled task with cron expression: {}", cronExpression);
        scheduledTask = taskScheduler.schedule(this::sendData, new CronTrigger(cronExpression));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public EntitlementUsage prepareUsageData() {
        EntitlementUsage usage = new EntitlementUsage();

        try(InputStream is = Files.newInputStream(licenseFile.getFile().toPath())) {
            usage.setAdditionalData(DigestUtils.md5Hex(is).getBytes());
        } catch (IOException e) {
            logger.warn("Could not calculate md5 for license file", e);
        }

        usage.setValidatorId(entitlementValidator.getId());
        usage.setClientId(entitlementValidator.getClientId());
        usage.setHostId(hostId);
        usage.setValidatorVersion(entitlementValidator.getVersion());
        usage.setModule(provider.getModule());
        usage.setModuleVersion(entitlementValidator.getPackageVersion());
        usage.setModuleBuild(entitlementValidator.getPackageBuild());
        usage.setEntitlements(provider.getCurrentUsage());

        usage.setNetworkInterfaces(getNetworkInterfaces());

        try {
            RuntimeMXBean runtime = ManagementFactory.getRuntimeMXBean();
            long startTime = runtime.getStartTime();
            long upTime = runtime.getUptime();

            usage.setCores(Runtime.getRuntime().availableProcessors());
            usage.setTotalJvmMemory(Runtime.getRuntime().totalMemory());
            usage.setFreeJvmMemory(Runtime.getRuntime().freeMemory());

            OperatingSystemMXBean os = (com.sun.management.OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();


            usage.setTotalPhysicalMemory(os.getTotalPhysicalMemorySize());
            usage.setFreePhysicalMemory(os.getFreePhysicalMemorySize());
            usage.setTotalSwapMemory(os.getTotalSwapSpaceSize());
            usage.setFreeSwapMemory(os.getFreeSwapSpaceSize());

            usage.setStartupDate(Instant.ofEpochMilli(startTime));
            usage.setRunDuration(upTime);
            usage.setOsName(os.getName());
            usage.setOsVersion(os.getVersion());
            usage.setOsArch(os.getArch());
        } catch (Exception e) {
            logger.info("Unknown error collecting licensing information", e);
        }

        return usage;
    }

    /**
     * Collects the information of all network interfaces available.
     * @return list of network interfaces
     */
    protected List<EntitlementUsage.NetworkInterface> getNetworkInterfaces() {
        List<EntitlementUsage.NetworkInterface> result = new LinkedList<>();
        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            while (interfaces.hasMoreElements()) {
                NetworkInterface ni = interfaces.nextElement();
                logger.debug("Checking network interface {}", ni.getName());
                Enumeration<InetAddress> addresses = ni.getInetAddresses();
                if(!ni.isLoopback() && addresses.hasMoreElements()) {
                    logger.debug("Network interface {} will be sent", ni.getName());
                    EntitlementUsage.NetworkInterface iface = new EntitlementUsage.NetworkInterface();
                    iface.setName(ni.getName());
                    InetAddress address = addresses.nextElement();
                    if (isNotEmpty(address.getHostAddress())) {
                        iface.setIpAddress(address.getHostAddress());
                    }
                    if (isNotEmpty(address.getHostName())) {
                        iface.setHostname(address.getHostName());
                    }
                    byte[] macAddress = ni.getHardwareAddress();
                    if (macAddress != null) {
                        StringBuilder sb = new StringBuilder();
                        for (byte b : macAddress) {
                            sb.append(String.format("%02x:", b));
                        }
                        sb.deleteCharAt(sb.length() - 1);
                        iface.setHardwareAddress(sb.toString());
                    }
                    result.add(iface);
                }
            }
        } catch (Exception e) {
            logger.info("Unknown error collecting licensing information", e);
        }
        return result;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void sendData() {
        if(!license.getConfiguration().isEnabled()) {
            return;
        }

        Path tempUsage = null;

        try {
            EntitlementUsage usage = prepareUsageData();
            tempUsage = Files.createTempFile("crafter", "lic-usage");

            ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
            try(OutputStream out = Files.newOutputStream(tempUsage)) {
                objectMapper.writeValue(out, usage);
                PGPUtils.encrypt(tempUsage, new ByteArrayInputStream(publicKeyContent), encrypted);
                byte[] content = encrypted.toByteArray();
                DatagramPacket packet = new DatagramPacket(content, content.length,
                    InetAddress.getByName(
                        license.getConfiguration().getDomain()), license.getConfiguration().getPort());
                socket.send(packet);
            }
        } catch (UnknownHostException e) {
            // hide the stacktrace
            logger.info("Error collecting licensing information");
        } catch (Exception e) {
            logger.info("Error collecting licensing information", e);
        } finally {
            if(tempUsage != null) {
                FileUtils.deleteQuietly(tempUsage.toFile());
            }
        }
    }

    @Override
    public void destroy() {
        sendData();
    }

}
