/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.manager.impl;

import java.util.zip.ZipFile;

import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.exception.LicenseNotFoundException;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.manager.LicenseUpdatedEvent;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.ApplicationEventPublisher;

/**
 * Default implementation of {@link LicenseManager}.
 *
 * @author joseross
 */
public class DefaultLicenseManagerImpl extends AbstractLicenseLoader implements LicenseManager {

    private static final Logger logger = LicenseManager.getLogger();

    /**
     * The last modified time of the license file.
     */
    protected long lastModified = 0;

    /**
     * Object used to publish license related events.
     */
    protected  ApplicationEventPublisher publisher;

    @Required
    public void setPublisher(final ApplicationEventPublisher publisher) {
        this.publisher = publisher;
    }

    public void init() throws Exception {
        checkLicenseFile(false);
    }

    protected void checkLicenseFile(boolean publishEvent) throws Exception {
        if(!licenseFile.exists()) {
            logger.error("License file not found, unable to start");
            throw new LicenseNotFoundException("License file not found");
        }
        if(licenseFile.lastModified() != lastModified) {
            logger.info("New license found, starting reload");
            try(ZipFile zip = new ZipFile(licenseFile.getFile())) {
                if(zip.getEntry(licenseName) == null || zip.getEntry(privateKeyName) == null ||
                    zip.getEntry(publicKeyName) == null) {
                    throw new IllegalStateException("License file is incomplete or corrupt");
                }
                lastModified = licenseFile.lastModified();
                if(publishEvent) {
                    publisher.publishEvent(new LicenseUpdatedEvent(this));
                }
            } catch (Exception e) {
                logger.error("License found but could not be loaded, unable to start", e);
                throw new InvalidLicenseException("License found but could not be loaded", e);
            }
        }

    }

    /**
     * {@inheritDoc}
     */
    public void loadLicense() {
        logger.debug("Checking license");
        try {
            checkLicenseFile(true);
        } catch (Exception e) {
            logger.warn("Error checking the license file", e);
        }
    }

}
