/*
 * Copyright (C) 2007-2019 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.validator.impl;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.jar.JarFile;

import org.apache.commons.lang3.StringUtils;
import org.craftercms.commons.entitlements.exception.EntitlementException;
import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.exception.LicenseNotFoundException;
import org.craftercms.commons.entitlements.manager.LicenseAware;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.manager.LicenseUpdatedEvent;
import org.craftercms.commons.entitlements.model.EntitlementType;
import org.craftercms.commons.entitlements.model.License;
import org.craftercms.commons.entitlements.model.LicenseType;
import org.craftercms.commons.entitlements.model.Module;
import org.craftercms.commons.entitlements.validator.EntitlementValidator;
import org.craftercms.commons.monitoring.VersionInfo;
import org.slf4j.Logger;

/**
 * Implementation of {@link EntitlementValidator} for Crafter Enterprise Edition.
 *
 * @author joseross
 */
public class DefaultEntitlementValidatorImpl extends AbstractLicenseLoader
    implements EntitlementValidator, LicenseAware {

    private static final Logger logger = LicenseManager.LOGGER;

    /**
     * Object loaded from the license file.
     */
    protected License license;

    /**
     * Performs an early validation of the license file.
     * @throws EntitlementException if the license file is not found or can't be read.
     */
    public void init() throws EntitlementException {
        if(new Date().after(LicenseManager.LIMIT)) {
            logger.info("Crafter CMS v{} Enterprise Edition", getPackageVersion());
            loadLicense();
        }
    }

    public synchronized void loadLicense() throws EntitlementException {
        logger.info("Loading license");
        if (!licenseFile.exists()) {
            logger.error("License file not found");
            throw new LicenseNotFoundException();
        }
        try {
            license = decryptLicense();
            if(license != null) {
                SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                format.setTimeZone(TimeZone.getTimeZone("UTC"));
                logger.info("Product licensed to: {}", license.getClientName());
                logger.info("License valid from {} through {}", format.format(license.getContractStartDate()),
                    format.format(license.getContractEndDate()));
                logger.info("License for {} environments", license.getEnvironment());
                validateLicense();
            }
        } catch (Exception e) {
            logger.error("License found but could not be loaded, unable to start", e);
            throw new InvalidLicenseException("License found but could not be loaded", e);
        }
    }

    /**
     * Performs general validations on the license object.
     */
    public synchronized void validateLicense() {
        if (new Date().after(LicenseManager.LIMIT)) {
            logger.debug("Validating current license");
            if (license.getLicenseType() == LicenseType.STANDARD && license.getContractEndDate().before(new Date())) {
                logger.error("Current license is expired");
                System.exit(0);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void validateEntitlement(Module module, EntitlementType entitlementType, int currentValue,
                                                 int newAmount) {
        // Actual validation will not be backported
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized long getId() {
        return license.getId();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized long getClientId() {
        return license.getClientId();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized String getVersion() {
        return license.getVersion();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized String getDescription() {
        // Not backported
        return StringUtils.EMPTY;
    }

    // Backport of default methods from interfaces

    /**
     * Provides the version of the containing JAR file.
     * @return the version
     */
    public String getPackageVersion() {
        try {
            VersionInfo versionMonitor = VersionInfo.getVersion(
                new JarFile(new File(getClass().getProtectionDomain().getCodeSource().getLocation().toURI()))
                    .getManifest());
            return versionMonitor.getPackageVersion();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Provides the build of the containing JAR file.
     * @return the build
     */
    public String getPackageBuild() {
        try {
            VersionInfo versionMonitor = VersionInfo.getVersion(
                new JarFile(new File(getClass().getProtectionDomain().getCodeSource().getLocation().toURI()))
                    .getManifest());
            return versionMonitor.getPackageBuild();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Performs all actions required when the license is updated.
     */
    public void reloadLicense() {
        try {
            loadLicense();
        } catch (EntitlementException e) {
            logger.error("Could not reload the license file", e);
        }
    }

    /**
     * Handles the license update event
     * @param event the event
     */
    public void onApplicationEvent(LicenseUpdatedEvent event) {
        reloadLicense();
    }

}