/*
 * Copyright (C) 2007-2018 Crafter Software Corporation. All Rights Reserved.
 *
 * This program is licensed under the Crafter Enterprise Software License Agreement, and its use is strictly limited
 * to operation with Crafter CMS Enterprise Edition. Unauthorized use, distribution, or modification is strictly
 * prohibited.
 *
 */

package org.craftercms.commons.entitlements.manager.impl;

import java.util.Date;
import java.util.zip.ZipFile;

import org.craftercms.commons.entitlements.exception.InvalidLicenseException;
import org.craftercms.commons.entitlements.exception.LicenseNotFoundException;
import org.craftercms.commons.entitlements.manager.LicenseManager;
import org.craftercms.commons.entitlements.manager.AbstractLicenseLoader;
import org.craftercms.commons.entitlements.manager.LicenseUpdatedEvent;
import org.slf4j.Logger;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * Default implementation of {@link LicenseManager}.
 *
 * @author joseross
 */
public class DefaultLicenseManagerImpl extends AbstractLicenseLoader implements LicenseManager,
    ApplicationContextAware {

    private static final Logger logger = LicenseManager.LOGGER;

    /**
     * The last modified time of the license file.
     */
    protected long lastModified = 0;

    protected ApplicationContext appContext;

    public void init() throws Exception {
        if(new Date().after(LicenseManager.LIMIT)) {
            checkLicenseFile(false);
        }
    }

    protected void checkLicenseFile(boolean publishEvent) throws Exception {
        if(!licenseFile.exists()) {
            logger.error("License file not found, unable to start");
            throw new LicenseNotFoundException("License file not found");
        }
        if(licenseFile.lastModified() != lastModified) {
            logger.info("New license found, starting reload");
            try(ZipFile zip = new ZipFile(licenseFile.getFile())) {
                if(zip.getEntry(licenseName) == null || zip.getEntry(privateKeyName) == null ||
                    zip.getEntry(publicKeyName) == null) {
                    throw new IllegalStateException("License file is incomplete or corrupt");
                }
                lastModified = licenseFile.lastModified();
                if(publishEvent) {
                    appContext.publishEvent(new LicenseUpdatedEvent(this));
                }
            } catch (Exception e) {
                logger.error("License found but could not be loaded, unable to start", e);
                throw new InvalidLicenseException("License found but could not be loaded", e);
            }
        }

    }

    /**
     * {@inheritDoc}
     */
    public void loadLicense() {
        if(new Date().after(LicenseManager.LIMIT)) {
            logger.debug("Checking license");
            try {
                checkLicenseFile(true);
            } catch (Exception e) {
                System.exit(0);
            }
        }
    }

    @Override
    public void setApplicationContext(final ApplicationContext applicationContext) throws BeansException {
        appContext = applicationContext;
    }

}
