/*
 * Decompiled with CFR 0.152.
 */
package org.cloudfoundry.multiapps.controller.persistence.services;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.io.FileUtils;
import org.cloudfoundry.multiapps.controller.persistence.model.FileEntry;
import org.cloudfoundry.multiapps.controller.persistence.model.ImmutableFileEntry;
import org.cloudfoundry.multiapps.controller.persistence.services.FileContentProcessor;
import org.cloudfoundry.multiapps.controller.persistence.services.FileStorage;
import org.cloudfoundry.multiapps.controller.persistence.services.FileStorageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemFileStorage
implements FileStorage {
    private static final String DEFAULT_FILES_STORAGE_PATH = "files";
    private final String storagePath;
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    public FileSystemFileStorage() {
        this(DEFAULT_FILES_STORAGE_PATH);
    }

    public FileSystemFileStorage(String storagePath) {
        this.storagePath = storagePath;
    }

    @Override
    public void addFile(FileEntry fileEntry, File file) throws FileStorageException {
        try (InputStream fileInputStream = Files.newInputStream(file.toPath(), new OpenOption[0]);){
            Path filesDirectory = this.getFilesDirectory(fileEntry.getSpace());
            Path newFilePath = Paths.get(filesDirectory.toString(), fileEntry.getId());
            this.logger.trace(MessageFormat.format("Storing file to path \"{0}\"...", newFilePath));
            Files.copy(fileInputStream, newFilePath, StandardCopyOption.REPLACE_EXISTING);
            File newFile = newFilePath.toFile();
            if (!newFile.exists()) {
                throw new FileStorageException(MessageFormat.format("Upload of file \"{0}\" to \"{1}\" failed", fileEntry.getName(), fileEntry.getNamespace()));
            }
            this.logger.debug(MessageFormat.format("Stored file \"{0}\" with size {1}", newFile, newFile.length()));
        }
        catch (IOException e) {
            throw new FileStorageException(e.getMessage(), e);
        }
    }

    @Override
    public List<FileEntry> getFileEntriesWithoutContent(List<FileEntry> fileEntries) throws FileStorageException {
        ArrayList<FileEntry> entriesWithoutContent = new ArrayList<FileEntry>();
        for (FileEntry entry : fileEntries) {
            if (this.hasContent(entry)) continue;
            entriesWithoutContent.add(entry);
        }
        return entriesWithoutContent;
    }

    @Override
    public void deleteFile(String id, String space) throws FileStorageException {
        try {
            Path filesDirectory = this.getFilesDirectory(space);
            Path filePath = Paths.get(filesDirectory.toString(), id);
            this.logger.debug(MessageFormat.format("Deleting file with path \"{0}\"...", filePath.toString()));
            Files.deleteIfExists(filePath);
        }
        catch (IOException e) {
            throw new FileStorageException(MessageFormat.format("Error deleting file with ID \"{0}\"", id), e);
        }
    }

    @Override
    public void deleteFilesBySpace(String space) throws FileStorageException {
        File spaceDirectory = this.getSpaceDirectory(space).toFile();
        try {
            if (spaceDirectory.exists()) {
                FileUtils.deleteDirectory((File)spaceDirectory);
            }
        }
        catch (IOException e) {
            throw new FileStorageException(MessageFormat.format("Error deleting directory \"{0}\"", spaceDirectory), e);
        }
    }

    @Override
    public void deleteFilesBySpaceAndNamespace(String space, String namespace) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int deleteFilesModifiedBefore(Date modificationTime) throws FileStorageException {
        final AtomicInteger deletedFiles = new AtomicInteger();
        final FileTime modificationTimeUpperBound = FileTime.fromMillis(modificationTime.getTime());
        try {
            Files.walkFileTree(Paths.get(this.storagePath, new String[0]), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (attrs.lastModifiedTime().compareTo(modificationTimeUpperBound) < 0) {
                        FileSystemFileStorage.this.logger.trace(MessageFormat.format("Deleting file with path \"{0}\"...", file.toString()));
                        boolean deleted = Files.deleteIfExists(file);
                        FileSystemFileStorage.this.logger.debug(MessageFormat.format("Deleted file with path \"{0}\": {1}", file.toString(), deleted));
                        if (deleted) {
                            deletedFiles.incrementAndGet();
                        }
                        return FileVisitResult.CONTINUE;
                    }
                    return super.visitFile(file, attrs);
                }
            });
        }
        catch (IOException e) {
            throw new FileStorageException(e.getMessage(), e);
        }
        return deletedFiles.intValue();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public <T> T processFileContent(String space, String id, FileContentProcessor<T> fileContentProcessor) throws FileStorageException {
        FileEntry fileEntry = this.createFileEntry(space, id);
        if (!this.hasContent(fileEntry)) {
            throw new FileStorageException(MessageFormat.format("File with ID \"{0}\" and space \"{1}\" does not exist.", fileEntry.getId(), fileEntry.getSpace()));
        }
        try (InputStream fileContentStream = this.getFileContentStream(fileEntry);){
            T t = fileContentProcessor.process(fileContentStream);
            return t;
        }
        catch (Exception e) {
            throw new FileStorageException(e);
        }
    }

    private FileEntry createFileEntry(String space, String id) {
        return ImmutableFileEntry.builder().space(space).id(id).build();
    }

    private InputStream getFileContentStream(FileEntry fileEntry) throws IOException {
        Path fileLocation = this.getFilePath(fileEntry);
        this.logger.trace(MessageFormat.format("Processing file \"{0}\"...", fileLocation));
        return Files.newInputStream(fileLocation, new OpenOption[0]);
    }

    public String getStoragePath() {
        return this.storagePath;
    }

    private Path getFilesDirectory(String space) throws IOException {
        Path filesPerSpaceDirectory = this.getFilesPerSpaceDirectory(space);
        Files.createDirectories(filesPerSpaceDirectory, new FileAttribute[0]);
        return filesPerSpaceDirectory;
    }

    private boolean hasContent(FileEntry entry) throws FileStorageException {
        try {
            Path filePath = this.getFilePath(entry);
            return filePath.toFile().exists();
        }
        catch (IOException e) {
            throw new FileStorageException(e.getMessage(), e);
        }
    }

    private Path getFilePath(FileEntry entry) throws IOException {
        Path filesDirectory = this.getFilesDirectory(entry.getSpace());
        return Paths.get(filesDirectory.toString(), entry.getId());
    }

    private Path getSpaceDirectory(String space) {
        return Paths.get(this.storagePath, space);
    }

    private Path getFilesPerSpaceDirectory(String space) {
        return Paths.get(this.storagePath, space, DEFAULT_FILES_STORAGE_PATH);
    }
}

