/*
 * Decompiled with CFR 0.152.
 */
package org.cloudfoundry.multiapps.controller.persistence.services;

import com.google.common.net.MediaType;
import java.io.File;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.cloudfoundry.multiapps.common.util.MiscUtil;
import org.cloudfoundry.multiapps.controller.persistence.model.FileEntry;
import org.cloudfoundry.multiapps.controller.persistence.model.ImmutableFileEntry;
import org.cloudfoundry.multiapps.controller.persistence.services.FileContentProcessor;
import org.cloudfoundry.multiapps.controller.persistence.services.FileStorage;
import org.cloudfoundry.multiapps.controller.persistence.services.FileStorageException;
import org.jclouds.blobstore.BlobStore;
import org.jclouds.blobstore.ContainerNotFoundException;
import org.jclouds.blobstore.domain.Blob;
import org.jclouds.blobstore.domain.StorageMetadata;
import org.jclouds.blobstore.options.ListContainerOptions;
import org.jclouds.blobstore.options.PutOptions;
import org.jclouds.http.HttpResponseException;
import org.jclouds.io.Payload;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

public class ObjectStoreFileStorage
implements FileStorage {
    private static final Logger LOGGER = LoggerFactory.getLogger(ObjectStoreFileStorage.class);
    private static final long RETRY_BASE_WAIT_TIME_IN_MILLIS = 5000L;
    private final BlobStore blobStore;
    private final String container;

    public ObjectStoreFileStorage(BlobStore blobStore, String container) {
        this.blobStore = blobStore;
        this.container = container;
    }

    @Override
    public void addFile(FileEntry fileEntry, File file) throws FileStorageException {
        String entryName = fileEntry.getId();
        long fileSize = fileEntry.getSize().longValue();
        Blob blob = this.blobStore.blobBuilder(entryName).payload(file).contentDisposition(fileEntry.getName()).contentType(MediaType.OCTET_STREAM.toString()).userMetadata(this.createFileEntryMetadata(fileEntry)).build();
        try {
            this.putBlobWithRetries(blob, 3);
            LOGGER.debug(MessageFormat.format("Stored file \"{0}\" with size {1}", fileEntry.getId(), fileSize));
        }
        catch (ContainerNotFoundException e) {
            throw new FileStorageException(MessageFormat.format("Upload of file \"{0}\" to \"{1}\" failed", fileEntry.getName(), fileEntry.getNamespace()));
        }
    }

    @Override
    public List<FileEntry> getFileEntriesWithoutContent(List<FileEntry> fileEntries) {
        Set existingFiles = this.blobStore.list(this.container).stream().map(StorageMetadata::getName).collect(Collectors.toSet());
        return fileEntries.stream().filter(fileEntry -> {
            String id = fileEntry.getId();
            return !existingFiles.contains(id);
        }).collect(Collectors.toList());
    }

    @Override
    public void deleteFile(String id, String space) {
        this.blobStore.removeBlob(this.container, id);
    }

    @Override
    public void deleteFilesBySpace(String space) {
        this.removeBlobsByFilter(blob -> this.filterBySpace((StorageMetadata)blob, space));
    }

    @Override
    public void deleteFilesBySpaceAndNamespace(String space, String namespace) {
        this.removeBlobsByFilter(blob -> this.filterBySpaceAndNamespace((StorageMetadata)blob, space, namespace));
    }

    @Override
    public int deleteFilesModifiedBefore(Date modificationTime) {
        return this.removeBlobsByFilter(blob -> this.filterByModificationTime((StorageMetadata)blob, modificationTime));
    }

    @Override
    public <T> T processFileContent(String space, String id, FileContentProcessor<T> fileContentProcessor) throws FileStorageException {
        FileEntry fileEntry = this.createFileEntry(space, id);
        try {
            Blob blob = this.getBlobWithRetries(fileEntry, 3);
            if (blob == null) {
                throw new FileStorageException(MessageFormat.format("File with ID \"{0}\" and space \"{1}\" does not exist.", fileEntry.getId(), fileEntry.getSpace()));
            }
            Payload payload = blob.getPayload();
            return this.processContent(fileContentProcessor, payload);
        }
        catch (Exception e) {
            throw new FileStorageException(e);
        }
    }

    private FileEntry createFileEntry(String space, String id) {
        return ImmutableFileEntry.builder().space(space).id(id).build();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> T processContent(FileContentProcessor<T> fileContentProcessor, Payload payload) throws FileStorageException {
        try (InputStream fileContentStream = payload.openStream();){
            T t = fileContentProcessor.process(fileContentStream);
            return t;
        }
        catch (Exception e) {
            throw new FileStorageException(e);
        }
    }

    private void putBlobWithRetries(Blob blob, int retries) {
        for (int i = 1; i <= retries; ++i) {
            try {
                this.blobStore.putBlob(this.container, blob, PutOptions.Builder.multipart());
                return;
            }
            catch (HttpResponseException e) {
                LOGGER.warn(MessageFormat.format("Attempt [{0}/{1}] to upload blob to ObjectStore failed with \"{2}\"", i, retries, e.getMessage()), (Throwable)e);
                if (i == retries) {
                    throw e;
                }
                MiscUtil.sleep((long)((long)i * this.getRetryWaitTime()));
                continue;
            }
        }
    }

    private Blob getBlobWithRetries(FileEntry fileEntry, int retries) {
        for (int i = 1; i <= retries; ++i) {
            Blob blob = this.blobStore.getBlob(this.container, fileEntry.getId());
            if (blob != null) {
                return blob;
            }
            LOGGER.warn(MessageFormat.format("Attempt [{0}/{1}] to download missing blob {2} from ObjectStore", i, retries, fileEntry.getId()));
            if (i == retries) break;
            MiscUtil.sleep((long)((long)i * this.getRetryWaitTime()));
        }
        return null;
    }

    protected long getRetryWaitTime() {
        return 5000L;
    }

    private Map<String, String> createFileEntryMetadata(FileEntry fileEntry) {
        HashMap<String, String> metadata = new HashMap<String, String>();
        metadata.put("SPACE".toLowerCase(), fileEntry.getSpace());
        metadata.put("MODIFIED".toLowerCase(), Long.toString(fileEntry.getModified().getTime()));
        if (fileEntry.getNamespace() != null) {
            metadata.put("NAMESPACE".toLowerCase(), fileEntry.getNamespace());
        }
        return metadata;
    }

    private int removeBlobsByFilter(Predicate<? super StorageMetadata> filter) {
        Set<String> entries = this.getEntryNames(filter);
        if (!entries.isEmpty()) {
            this.blobStore.removeBlobs(this.container, entries);
        }
        return entries.size();
    }

    private Set<String> getEntryNames(Predicate<? super StorageMetadata> filter) {
        return this.blobStore.list(this.container, new ListContainerOptions().withDetails()).stream().filter(Objects::nonNull).filter(filter).map(StorageMetadata::getName).collect(Collectors.toSet());
    }

    private boolean filterByModificationTime(StorageMetadata blobMetadata, Date modificationTime) {
        Map userMetadata = blobMetadata.getUserMetadata();
        if (CollectionUtils.isEmpty((Map)userMetadata)) {
            return true;
        }
        String longString = (String)userMetadata.get("MODIFIED".toLowerCase());
        try {
            long dateLong = Long.parseLong(longString);
            Date date = new Date(dateLong);
            return date.before(modificationTime);
        }
        catch (NumberFormatException e) {
            return true;
        }
    }

    private boolean filterBySpace(StorageMetadata blobMetadata, String space) {
        Map userMetadata = blobMetadata.getUserMetadata();
        if (CollectionUtils.isEmpty((Map)userMetadata)) {
            return false;
        }
        String spaceParameter = (String)userMetadata.get("SPACE".toLowerCase());
        return space.equals(spaceParameter);
    }

    private boolean filterBySpaceAndNamespace(StorageMetadata blobMetadata, String space, String namespace) {
        Map userMetadata = blobMetadata.getUserMetadata();
        if (CollectionUtils.isEmpty((Map)userMetadata)) {
            return false;
        }
        String spaceParameter = (String)userMetadata.get("SPACE".toLowerCase());
        String namespaceParameter = (String)userMetadata.get("NAMESPACE".toLowerCase());
        return space.equals(spaceParameter) && namespace.equals(namespaceParameter);
    }
}

