package tw.gov.president.manager.submanager.logmoniter

import android.annotation.SuppressLint
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import okhttp3.Interceptor
import okhttp3.MediaType
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.RequestBody
import okhttp3.internal.checkOffsetAndCount
import okhttp3.logging.HttpLoggingInterceptor
import okio.BufferedSink
import org.koin.core.inject
import retrofit2.HttpException
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import timber.log.Timber
import tw.gov.president.general.data.info.request.DeviceEventRequest
import tw.gov.president.general.data.loglevel.LogLevel
import tw.gov.president.manager.BaseDomainManager
import tw.gov.president.manager.BaseManagerData
import tw.gov.president.manager.submanager.logmoniter.api.LogApiRepository
import tw.gov.president.manager.submanager.logmoniter.api.LogApiService
import tw.gov.president.manager.submanager.logmoniter.api.LogAuthInterceptor
import tw.gov.president.provider.device.info.DeviceInfoProvider
import tw.gov.president.utils.general.app.GeneralUtils
import tw.gov.president.utils.general.utils.api.result.ApiResult
import tw.gov.president.utils.secret.AESUtils
import java.math.BigInteger
import java.nio.charset.Charset
import java.security.MessageDigest
import java.util.*
import java.util.concurrent.TimeUnit

class LogDomainManager(private val userid: String = "") : BaseDomainManager() {
    override val projectId: String = LOG_SERVER_PROJECT_ID
    override val defaultDomain: String = "moshiqhwh.com"
    private val interceptor: LogAuthInterceptor by inject()
    private val httpLoggingInterceptor: HttpLoggingInterceptor by inject()
    private val okHttpClient: OkHttpClient = loginOkHttpClient(interceptor, httpLoggingInterceptor)
    private val deviceManager: DeviceInfoProvider by inject()

    fun getLogApiRepository(domainType: DomainType = DomainType.API): LogApiRepository {
        val apiService = Retrofit.Builder()
            .addConverterFactory(GsonConverterFactory.create(gson))
            .client(okHttpClient)
            .baseUrl(getApiDomain(domainType))
            .build()
            .create(LogApiService::class.java)
        return LogApiRepository(
            apiService
        )
    }

    override fun getLibEnv(): String {
        Timber.i("FLAVOR= ${checkFlavor()}")
        return when (checkFlavor()) {
            FLAVOR_DEV -> "d"
            FLAVOR_SIT -> "p"
            else -> "p"
        }
    }

    private fun loginOkHttpClient(
        interceptor: Interceptor,
        httpLoggingInterceptor: HttpLoggingInterceptor
    ): OkHttpClient {
        val builder = OkHttpClient.Builder()
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(60, TimeUnit.SECONDS)
            .writeTimeout(60, TimeUnit.SECONDS)
            .addInterceptor(interceptor)
            .addInterceptor(httpLoggingInterceptor)
//        if (BaseManagerData.configData?.debug == true) {
//            builder.addNetworkInterceptor(StethoInterceptor())
//        }
        return builder.build()
    }

    @SuppressLint("MissingPermission")
    fun sendCrashReportFlow(project: String, message: String) =
        sendByType(project, LogLevel.ERROR, message)

    @SuppressLint("MissingPermission")
    fun sendLogFlow(project: String, message: String) = sendByType(project, LogLevel.INFO, message)

    @SuppressLint("MissingPermission")
    fun sendDeviceInfoKibanaFlow(project: String, message: String) =
        sendByType(project, LogLevel.DEVICE_INFO, message)

    @SuppressLint("MissingPermission")
    fun sendByType(project: String, level: LogLevel, message: String) =
        sendByType(project, level, userid, projectId, message)

    @SuppressLint("MissingPermission")
    fun sendByType(
        project: String,
        level: LogLevel,
        current_userId: String,
        project_id: String,
        message: String
    ) =
        flow {
            val devicedata = deviceManager.fetchDeviceData()
            if (message.isNullOrEmpty() || (level.getEvent("").contains("device_info"))) {

            }
            val request = deviceManager.getMobileInfo(
                BaseManagerData.configData?.appVersionCode ?: "",
                current_userId, project_id,
                goLibVersion, level.getEvent(project), level, message
            )
            if ((level.getEvent(project).contains("device_info"))) {
                if (request.data != null) {
                    request.data = message + "," + devicedata
                } else {
                    request.data = devicedata
                }
            }
            if (message.isNullOrEmpty()) {
                request.data = devicedata
            }
            if (BaseManagerData.configData?.debug == true) {
                Timber.w("Show Body :${gson.toJson(request)}   ")
            }
            val result = getLogApiRepository().sendLog(
                getHeaders(request), buildTextRequestBody(
                    buildEncryptData(
                        request
                    )
                )
            )
            if (!result.isSuccessful) {
                Timber.d("sendByType failed:${HttpException(result)}   ")
                throw HttpException(result)
            }
            emit(ApiResult.success(result))
        }
            .flowOn(Dispatchers.IO)
            .catch { e -> emit(ApiResult.error(e)) }

    private fun buildEncryptData(item: Any): String {
        return AESUtils.encryptAes(gson.toJson(item))
    }

    private fun buildTextRequestBody(data: String): RequestBody {
        return data.toRequestBody(MEDIA_TYPE_TEXT.toMediaTypeOrNull())
    }

    fun getHeaders(request: DeviceEventRequest): Map<String, String> {
        val full_app_id = BaseManagerData.configData!!.applicationId
        val full_device_id = GeneralUtils.getAndroidID()
        val full_device_version_code = BaseManagerData.configData!!.appVersionCode
        val full_request_id = UUID.randomUUID().toString()
        val timestamp = (System.currentTimeMillis() / 1000).toString()
        val length = full_device_id.length
        val jsonStr = gson.toJson(request)
        val raw = StringBuilder(full_app_id)
            .append(full_device_id.run { substring(length - 9) })
            .append(full_device_version_code)
            .append(full_request_id.substring(0, 5))
            .append(timestamp)
            .append(jsonStr)
            .toString()
        val auth = StringBuilder(BEARER_PREFIX).append(raw.toMD5()).toString()
        val headers = hashMapOf<String, String>()
        headers[AUTHORIZATION] = auth
        headers[APP_ID] = full_app_id
        headers[DEVICE_ID] = full_device_id
        headers[APP_VERSION_CODE] = full_device_version_code
        headers[REQUEST_ID] = full_request_id
        headers[TIMESTAMP] = timestamp
        headers[SIGN] = "auth $full_request_id"
        return headers
    }

    //FIXME TEMP
    private fun String.toMD5(): String {
        val md = MessageDigest.getInstance("MD5")
        return BigInteger(1, md.digest(toByteArray())).toString(16).padStart(32, '0')
    }

    private fun String.toRequestBody(contentType: MediaType? = null): RequestBody {
        var charset: Charset = Charsets.UTF_8
        var finalContentType: MediaType? = contentType
        if (contentType != null) {
            val resolvedCharset = contentType.charset()
            if (resolvedCharset == null) {
                charset = Charsets.UTF_8
                finalContentType = "$contentType; charset=utf-8".toMediaTypeOrNull()
            } else {
                charset = resolvedCharset
            }
        }
        val bytes = toByteArray(charset)
        return bytes.toRequestBody(finalContentType, 0, bytes.size)
    }

    private fun String.toMediaTypeOrNull(): MediaType? {
        return try {
            this.toMediaType()
        } catch (_: IllegalArgumentException) {
            null
        }
    }

    private fun ByteArray.toRequestBody(
        contentType: MediaType? = null,
        offset: Int = 0,
        byteCount: Int = size
    ): RequestBody {
        checkOffsetAndCount(size.toLong(), offset.toLong(), byteCount.toLong())
        return object : RequestBody() {
            override fun contentType() = contentType

            override fun contentLength() = byteCount.toLong()

            override fun writeTo(sink: BufferedSink) {
                sink.write(this@toRequestBody, offset, byteCount)
            }
        }
    }
}