package tw.gov.president.provider.device.info

import android.Manifest.permission.READ_PHONE_STATE
import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Context.*
import android.content.Intent
import android.content.IntentFilter
import android.hardware.Sensor
import android.hardware.SensorManager
import android.location.LocationManager
import android.net.wifi.WifiManager
import android.os.Build
import android.telephony.*
import android.telephony.cdma.CdmaCellLocation
import android.telephony.gsm.GsmCellLocation
import android.text.TextUtils
import androidx.annotation.RequiresPermission
import com.blankj.utilcode.util.DeviceUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.NetworkUtils
import com.blankj.utilcode.util.PhoneUtils
import com.github.florent37.application.provider.ApplicationProvider
import org.koin.core.KoinComponent
import timber.log.Timber
import tw.gov.president.general.data.info.DeviceData
import tw.gov.president.general.data.info.LogStringItem
import tw.gov.president.general.data.info.device.*
import tw.gov.president.general.data.info.request.DeviceEventRequest
import tw.gov.president.general.data.loglevel.LogLevel
import tw.gov.president.manager.BaseManagerData
import tw.gov.president.utils.general.app.GeneralUtils

class DeviceInfoProvider(private val userid: String = "") : KoinComponent {
    companion object {
        const val EVENT_DEVICE_INFO = "Device Info"
        const val PLATFORM_ANDROID = "Android"
        const val CELL_INFO_LTE = "CellInfoLte"
        const val CELL_INFO_CDMA = "CellInfoCdma"
        const val CELL_INFO_GSM = "CellInfoGsm"
        const val CDMA_MCC = "460"
        const val CDMA_MNC = "0"
        const val CDMA_LAC = "0"
        const val SIGN_LOCATION = ","
    }

    private val canNotGetInfo = "can not get info"

    val App = ApplicationProvider.application
    private var wifiInfoList = arrayListOf<WifiInfo>()
    private val gsmLocList = arrayListOf<GsmLoc>()
    private val cdmaLocList = arrayListOf<CdmaLoc>()
    private val gsmSignalList = arrayListOf<GsmSignal>()
    private val cdmaSignalList = arrayListOf<CdmaSignal>()
    private val evdoSignalList = arrayListOf<EvdoSignal>()
    private val wifiManager = App?.getSystemService(WIFI_SERVICE) as WifiManager
    private val telephonyManager = App?.getSystemService(TELEPHONY_SERVICE) as TelephonyManager
    private val locationManager = App?.getSystemService(LOCATION_SERVICE) as LocationManager
    private val sensorManager = App?.getSystemService(SENSOR_SERVICE) as SensorManager

    private var isRegisterListener = false

    public fun getMobileInfo(
        version_name: String,
        userId: String,
        domain_version: String,
        event: String,
        logLevel: LogLevel,
        data: String
    ): DeviceEventRequest {
        return DeviceEventRequest(
            logLevel = logLevel.value.toLong(),
            osVersion = Build.VERSION.SDK_INT.toString(),
            appVersion = version_name,
            deviceId = GeneralUtils.getAndroidID(),
            userId = userId,
            deviceTime = System.currentTimeMillis() / 1000,
            event = event,
            env = BaseManagerData.configData?.flavor ?: "",
            platform = PLATFORM_ANDROID,
            goLibVersion = domain_version,
            data = LogStringItem(data)
        )
    }

    public fun fetchDeviceData(): DeviceData {
        return DeviceData(
            getDeviceInfo(),
            getNetworkInfo(),
            getPhoneInfo(),
            getGsmInfo(),
            wifiInfoList,
            gsmLocList,
            cdmaLocList,
            gsmSignalList,
            cdmaSignalList,
            evdoSignalList,
            getSensorInfo()
        )
    }

    fun registerListener() {
        if (!isRegisterListener) {
            isRegisterListener = true
            try {
//                telephonyManager.listen(phoneStateListener, PhoneStateListener.LISTEN_CELL_INFO)
            } catch (e: Exception) {
                Timber.e("LISTEN_CELL_INFO error = $e")
            }
//            try {
//                telephonyManager.listen(phoneStateListener, PhoneStateListener.LISTEN_CELL_LOCATION)
//            } catch (e: Exception) {
//                Timber.e("LISTEN_CELL_LOCATION error = $e")
//            }
//            try {
//                telephonyManager.listen(
//                    phoneStateListener,
//                    PhoneStateListener.LISTEN_SIGNAL_STRENGTHS
//                )
//            } catch (e: Exception) {
//                Timber.e("LISTEN_SIGNAL_STRENGTHS error = $e")
//            }
            val intentFilter = IntentFilter()
            intentFilter.addAction(WifiManager.SCAN_RESULTS_AVAILABLE_ACTION)
            App?.registerReceiver(wifiScanReceiver, intentFilter)
            startWifiScan()
        }
    }

    fun unregisterListener() {
        if (isRegisterListener) {
            isRegisterListener = false
//            telephonyManager.listen(phoneStateListener, PhoneStateListener.LISTEN_NONE)
            App?.unregisterReceiver(wifiScanReceiver)
        }
    }

//    private val phoneStateListener = object : PhoneStateListener() {
//        override fun onCellLocationChanged(location: CellLocation?) {
//            super.onCellLocationChanged(location)
//            when (location) {
//                is GsmCellLocation -> {
//                    if (gsmLocList.size < 11) {
//                        Timber.d("Location: $location")
//                        val gsmLoc = GsmLoc(location.lac, location.cid, location.psc)
//                        gsmLocList.add(gsmLoc)
//                    }
//                }
////                is CdmaCellLocation -> {
////                    if (cdmaLocList.size < 11) {
////                        Timber.d("Location: $location")
////                        val cdmaLoc = CdmaLoc(
////                            location.baseStationId,
////                            location.baseStationLatitude,
////                            location.baseStationLongitude,
////                            location.networkId,
////                            location.systemId
////                        )
////                        cdmaLocList.add(cdmaLoc)
////                    }
////                }
//            }
//        }
//
//        override fun onSignalStrengthsChanged(signalStrength: SignalStrength) {
//            super.onSignalStrengthsChanged(signalStrength)
//            Timber.d("onSignalStrengthsChanged: $signalStrength")
//            when {
//                signalStrength.isGsm -> {
//                    val gsmSignal =
//                        GsmSignal(signalStrength.gsmBitErrorRate, signalStrength.gsmSignalStrength)
//                    gsmSignalList.add(gsmSignal)
//                }
//                signalStrength.cdmaDbm > 0 -> {
//                    val cdmaSignal = CdmaSignal(signalStrength.cdmaDbm, signalStrength.cdmaEcio)
//                    cdmaSignalList.add(cdmaSignal)
//                }
//                else -> {
//                    val evdoSignal = EvdoSignal(
//                        signalStrength.evdoDbm,
//                        signalStrength.evdoEcio,
//                        signalStrength.evdoSnr
//                    )
//                    evdoSignalList.add(evdoSignal)
//                }
//            }
//        }
//    }

    private fun startWifiScan() {
        if (!wifiManager.isWifiEnabled) {
            wifiManager.isWifiEnabled = true
        }
        wifiManager.startScan()
    }

    private val wifiScanReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {
            val success = intent.getBooleanExtra(WifiManager.EXTRA_RESULTS_UPDATED, false)
            if (success) {
                wifiInfoList = scanSuccess()
            }
        }
    }

    private fun scanSuccess(): ArrayList<WifiInfo> {
        val wifiInfoList = arrayListOf<WifiInfo>()
        val results = wifiManager.scanResults
        results.forEach {
            val wifiInfo = WifiInfo(
                it.SSID,
                it.BSSID,
                it.capabilities,
                it.centerFreq0,
                it.centerFreq1,
                it.channelWidth,
                it.frequency,
                it.level,
                it.is80211mcResponder,
                it.isPasspointNetwork,
                it.operatorFriendlyName.toString(),
                it.venueName.toString(),
                it.timestamp,
                it.describeContents()
            )
            wifiInfoList.add(wifiInfo)
        }
        return wifiInfoList
    }

    private fun getDeviceInfo(): DeviceInfo {
        return DeviceInfo(
            DeviceUtils.getSDKVersionCode(),
            DeviceUtils.getSDKVersionName(),
            DeviceUtils.getAndroidID(),
            DeviceUtils.getMacAddress(),
            DeviceUtils.getManufacturer(),
            DeviceUtils.getModel(),
            DeviceUtils.getABIs().toMutableList()
        )
    }

    private fun getNetworkInfo(): NetworkInfo {
        return NetworkInfo(
            NetworkUtils.is4G(),
            NetworkUtils.isMobileData(),
            NetworkUtils.isConnected(),
            NetworkUtils.isAvailableByPing(),
            NetworkUtils.isWifiConnected(),
            NetworkUtils.isWifiAvailable(),
            NetworkUtils.getNetworkOperatorName(),
            NetworkUtils.getNetworkType(),
            NetworkUtils.getIPAddress(true),
            NetworkUtils.getIPAddress(false),
            NetworkUtils.getIpAddressByWifi(),
            NetworkUtils.getGatewayByWifi(),
            NetworkUtils.getNetMaskByWifi(),
            NetworkUtils.getServerAddressByWifi()
        )
    }

    @SuppressLint("MissingPermission")
    private fun getPhoneInfo(): PhoneInfo {
        val deviceId =
            try {
                PhoneUtils.getDeviceId()
            } catch (e: Exception) {
                Timber.e("deviceId error =${e}")
                e.message
            }
        val serial = try {
            PhoneUtils.getSerial()
        } catch (e: Exception) {
            Timber.e("serial error =${e}")
            e.message
        }
        val imei = try {
            PhoneUtils.getIMEI()
        } catch (e: Exception) {
            Timber.e("imei error =${e}")
            e.message
        }

        val meid = try {
            PhoneUtils.getMEID()
        } catch (e: Exception) {
            Timber.e("meid error =${e}")
            e.message
        }
        val imsi = try {
            PhoneUtils.getIMSI()
        } catch (e: Exception) {
            Timber.e("imsi error =${e}")
            e.message
        }
        val phoneType = try {
            PhoneUtils.getPhoneType().toString()
        } catch (e: Exception) {
            Timber.e("phoneType error =${e}")
            e.message
        }
        val isSimCardReady = try {
            PhoneUtils.isSimCardReady().toString()
        } catch (e: Exception) {
            Timber.e("isSimCardReady error =${e}")
            e.message
        }
        val getSimOperatorName = try {
            PhoneUtils.getSimOperatorName()
        } catch (e: Exception) {
            Timber.e("getSimOperatorName error =${e}")
            e.message
        }
        val getSimOperatorByMnc = try {
            PhoneUtils.getSimOperatorByMnc()
        } catch (e: Exception) {
            Timber.e("getSimOperatorByMnc error =${e}")
            e.message
        }
        val getPhoneStatus = try {
            getPhoneStatus().toString()
        } catch (e: Exception) {
            Timber.e("getPhoneStatus error =${e}")
            e.message
        }

        val result = PhoneInfo(
            if (!TextUtils.isEmpty(deviceId)) deviceId else canNotGetInfo,
            if (!TextUtils.isEmpty(serial)) serial else canNotGetInfo,
            if (!TextUtils.isEmpty(imei)) imei else canNotGetInfo,
            if (!TextUtils.isEmpty(meid)) meid else canNotGetInfo,
            if (!TextUtils.isEmpty(imsi)) imsi else canNotGetInfo,
            phoneType,
            isSimCardReady,
            if (!TextUtils.isEmpty(getSimOperatorName)) getSimOperatorName else canNotGetInfo,
            if (!TextUtils.isEmpty(getSimOperatorByMnc)) getSimOperatorByMnc else canNotGetInfo,
            getPhoneStatus
        )

        Timber.i("getPhoneInfo result = ${GsonUtils.toJson(result)}")
        return result

    }

    @SuppressLint("HardwareIds")
    @RequiresPermission(READ_PHONE_STATE)
    private fun getPhoneStatus(): ArrayList<PhoneStatus> {
        val phoneStatus = PhoneStatus(
            if (!TextUtils.isEmpty(telephonyManager.deviceId)) telephonyManager.deviceId else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.deviceSoftwareVersion)) telephonyManager.deviceSoftwareVersion else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.line1Number)) telephonyManager.line1Number else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.networkCountryIso)) telephonyManager.networkCountryIso else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.networkOperator)) telephonyManager.networkOperator else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.networkOperatorName)) telephonyManager.networkOperatorName else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.networkType.toString())) telephonyManager.networkType.toString() else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.phoneType.toString())) telephonyManager.phoneType.toString() else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.simCountryIso)) telephonyManager.simCountryIso else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.simOperator)) telephonyManager.simOperator else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.simOperatorName)) telephonyManager.simOperatorName else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.simSerialNumber)) telephonyManager.simSerialNumber else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.simState.toString())) telephonyManager.simState.toString() else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.subscriberId)) telephonyManager.subscriberId else canNotGetInfo,
            if (!TextUtils.isEmpty(telephonyManager.voiceMailNumber)) telephonyManager.voiceMailNumber else canNotGetInfo
        )
        return arrayListOf(phoneStatus)
    }

    @SuppressLint("MissingPermission")
    private fun getGsmInfo(): ArrayList<GsmInfo> {
        val gsmInfoList = arrayListOf<GsmInfo>()
        var allCellInfo: List<CellInfo>? = null
        try {
            allCellInfo = telephonyManager.allCellInfo
        } catch (e: java.lang.Exception) {
            Timber.e("getGsmInfo = ${e}")
        }
        if (allCellInfo == null || allCellInfo.isEmpty()) return gsmInfoList

        allCellInfo.forEach {
            when (it.toString().split(":")[0]) {
                CELL_INFO_LTE -> {
                    val cellInfoLte = it as CellInfoLte
                    val cellIdentity = cellInfoLte.cellIdentity
                    val gsmInfo = GsmInfo(
                        cellIdentity.mcc.toString(),
                        cellIdentity.mnc.toString(),
                        cellIdentity.tac.toString(),
                        cellIdentity.ci.toString(),
                        (-131 + 2 * cellInfoLte.cellSignalStrength.dbm).toString()
                    )
                    gsmInfoList.add(gsmInfo)
                }
                CELL_INFO_CDMA -> {
                    val cellInfoCdma = it as CellInfoCdma
                    val cellIdentity = cellInfoCdma.cellIdentity
                    val gsmInfo = GsmInfo(
                        CDMA_MCC,
                        CDMA_MNC,
                        CDMA_LAC,
                        cellIdentity.basestationId.toString(),
                        cellInfoCdma.cellSignalStrength.dbm.toString()
                    )
                    gsmInfoList.add(gsmInfo)
                }
                CELL_INFO_GSM -> {
                    val cellInfoGsm = it as CellInfoGsm
                    val cellIdentity = cellInfoGsm.cellIdentity
                    val gsmInfo = GsmInfo(
                        cellIdentity.mcc.toString(),
                        cellIdentity.mnc.toString(),
                        cellIdentity.lac.toString(),
                        cellIdentity.cid.toString(),
                        cellInfoGsm.cellSignalStrength.dbm.toString()
                    )
                    gsmInfoList.add(gsmInfo)
                }
            }
        }
        return gsmInfoList
    }

    private fun getSensorInfo(): ArrayList<SensorInfo> {
        val sensorInfoList = arrayListOf<SensorInfo>()
        val sensorList = sensorManager.getSensorList(Sensor.TYPE_ALL)
        if (sensorList == null || sensorList.isEmpty()) return sensorInfoList
        sensorList.forEach { sensor ->
            val sensorInfo = SensorInfo(sensor.name, sensor.version, sensor.vendor)
            sensorInfoList.add(sensorInfo)
        }
        return sensorInfoList
    }
}
