/*
 * Copyright 2014 The Vibe Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.atmosphere.vibe.server;

import java.io.Serializable;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

import org.atmosphere.vibe.platform.Action;
import org.atmosphere.vibe.platform.Actions;
import org.atmosphere.vibe.platform.ConcurrentActions;

/**
 * {@link Server} implementation for clustering.
 * <p>
 * With this implementation, {@code server.all(action)} have {@code action} be
 * executed with every socket in every server in the cluster.
 * <p>
 * This implementation adopts the publish and subscribe model from Java Message
 * Service to support clustering. Here, the exchanged message represents method
 * invocation to be executed by every server in the cluster. The following
 * methods create such messages.
 * <ul>
 * <li>{@link Server#all())}</li>
 * <li>{@link Server#all(Action)}</li>
 * <li>{@link Server#byId(String)}</li>
 * <li>{@link Server#byId(String, Action)}</li>
 * <li>{@link Server#byTag(String...))}</li>
 * <li>{@link Server#byTag(String, Action)))}</li>
 * <li>{@link Server#byTag(String[], Action)}</li>
 * </ul>
 * A message created by this server is passed to
 * {@link ClusteredServer#publishAction(Action)} and a message created by other
 * servers is expected to be passed to {@link ClusteredServer#messageAction()}.
 * Therefore, what you need to do is to publish a message given through
 * {@link ClusteredServer#publishAction(Action)} to every server in the cluster
 * and to subscribe a published message by other servers to delegate it to
 * {@link ClusteredServer#messageAction()}.
 * <p>
 * Accordingly, such message must be able to be serialized and you have to pass
 * {@link Action} implementing {@link Serializable}. However, serialization of
 * inner classes doesn't work in some cases as expected so that always use
 * {@link Sentence} instead of action if possible unless you use lambda
 * expressions.
 * 
 * @author Donghwan Kim
 * @see <a
 *      href="http://docs.oracle.com/javase/7/docs/platform/serialization/spec/serial-arch.html#4539">Note
 *      of the Serializable Interface</a>
 */
public class ClusteredServer extends DefaultServer {

    private Actions<Map<String, Object>> publishActions = new ConcurrentActions<>();
    private Action<Map<String, Object>> messageAction = new Action<Map<String, Object>>() {
        @SuppressWarnings("unchecked")
        @Override
        public void on(Map<String, Object> map) {
            String methodName = (String) map.get("method");
            Object[] args = (Object[]) map.get("args");
            switch (methodName) {
            case "all":
                ClusteredServer.super.all((Action<ServerSocket>) args[0]);
                break;
            case "byId":
                ClusteredServer.super.byId((String) args[0], (Action<ServerSocket>) args[1]);
                break;
            case "byTag":
                ClusteredServer.super.byTag((String[]) args[0], (Action<ServerSocket>) args[1]);
                break;
            default:
                throw new IllegalArgumentException("Illegal method name in processing message: "
                        + methodName);
            }
        }
    };

    @Override
    public Server all(Action<ServerSocket> action) {
        publishMessage("all", action);
        return this;
    }

    @Override
    public Server byId(String id, Action<ServerSocket> action) {
        publishMessage("byId", id, action);
        return this;
    }

    @Override
    public Server byTag(String[] names, Action<ServerSocket> action) {
        publishMessage("byTag", names, action);
        return this;
    }

    private void publishMessage(String method, Object... args) {
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("method", method);
        map.put("args", args);
        publishActions.fire(Collections.unmodifiableMap(map));
    }

    /**
     * Adds an action to be called with a message to be published to every node
     * in the cluster.
     */
    public Server publishAction(Action<Map<String, Object>> action) {
        publishActions.add(action);
        return this;
    }

    /**
     * An action to receive a message published from one of nodes in the
     * cluster.
     */
    public Action<Map<String, Object>> messageAction() {
        return messageAction;
    }

}
