/*
 * Copyright 2005  The Apache Software Foundation
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.ws.jaxme.maven.plugins;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.maven.model.Resource;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.apache.ws.jaxme.generator.Generator;
import org.apache.ws.jaxme.generator.SchemaReader;
import org.apache.ws.jaxme.generator.impl.GeneratorImpl;
import org.apache.ws.jaxme.generator.sg.SGFactoryChain;
import org.apache.ws.jaxme.generator.sg.impl.JAXBSchemaReader;
import org.apache.ws.jaxme.generator.sg.impl.JaxMeSchemaReader;
import org.apache.ws.jaxme.logging.Logger;
import org.apache.ws.jaxme.logging.LoggerAccess;
import org.apache.ws.jaxme.logging.LoggerFactory;
import org.codehaus.plexus.util.DirectoryScanner;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


/** This is the "jaxme:jaxme" goal. It runs the source
 * generator.
 * @goal jaxme
 * @phase generate-sources
 * @description Runs the JaxMe binding compiler
 * and generates the source files.
 * @requiresDependencyResolution test
 */
public class JaxMeGoal extends AbstractMojo {
	/** This property specifies a set of external
	 * binding files, which are being applied.
	 * @parameter
	 */
	private String[] bindings;

	/** Specifies classpath elements, which should be added to
	 * the plugins classpath.
	 * @parameter expression="${project.compileClasspathElements}"
	 * @required
	 * @readonly
	 */
	private List classpathElements;

	/** This property specifies a set of files, which
	 * are referenced from within the XML schema files,
	 * but aren't processed as XML schema files for
	 * themselves. Specifying files in the "depends" set
	 * allows to include them into the uptodate check.
	 * @parameter
	 */
	private String[] depends;

	/** Setting this property to true enables JaxMe's extension
	 * mode. Several vendor extensions are available, including
	 * some extensions, which are originally specified by the
	 * JAXB reference implementation.
	 * @parameter expression="false"
	 */
	private boolean extension;

	/** Setting this property to suppresses the builtin
	 * uptodate check.
	 * @parameter expression="false"
	 */
	private boolean force;

	/** Sets the Java package name.
	 * @parameter
	 */
	private String packageName;

	/** This property specifies a set of files, which are
	 * being produced by the source generator. Any element
	 * in the set may contain wildcards.
	 * The set of produced files is used for checking,
	 * whether the generated files are uptodate. If that is
	 * the case, then the generator will omit recreating
	 * them. If the uptodate check fails, or if the "force"
	 * property is turned on, then the source generator
	 * will actually be invoked. In that case, the set of
	 * produces files will also be used for removing old
	 * files, if the "removeOldOutput" feature is turned
	 * on.
	 * @parameter
	 */
	private String[] produces;

	/** The Maven project.
	 * @parameter expression="${project}"
	 * @required
	 * @readonly
	 */
	private MavenProject project;

	/** The properties being set on the generator.
	 * @parameter
	 */
	private Map properties;

	/** Setting this property to true will make the generated
	 * files read-only.
	 * @parameter expression="false"
	 */
	private boolean readOnly;

	/** If this property is set to true, and one or more
	 * produces "produces" elements are specified, then the plugin
	 * will remove all files matching the "produces" elements
	 * before running the source generator.
	 * @parameter expression="false"
	 */
	private boolean removingOldOutput;

	/** Sets the schema reader being used. The default
	 * schema reader is an instance of {@link JAXBSchemaReader}.
	 * The default changes, if "extension" is set to true,
	 * in which case the {@link JaxMeSchemaReader} is being
	 * used.
	 * @parameter
	 */
	private String schemaReader;
	
	/** The set of schemas being compiled. Schema names may
	 * include wildcards.
	 * @parameter
	 */
	private String[] schemas;

	/** The set of factory chains, which are being added to the
	 * generator. Factory chains are modifying the generators
	 * behaviour. A good example is the
	 * <code>org.apache.ws.jaxme.pm.generator.jdbc.JaxMeJdbcSG</code>.
	 * @parameter
	 */
	private String[] sgFactoryChain;

	/** The target directory for source files. Defaults to
	 * @parameter expression="${project.build.directory}/jaxme/java"
	 */
	private String srcTarget;

	/** The target directory for resource files. Defaults to
	 * @parameter expression="${project.build.directory}/jaxme/resources"
	 */
	private String resourceTarget;

	/** Setting this property to true advices the plugin to use
	 * a validating XML parser for reading the schema files.
	 * By default, validation is disabled.
	 * @parameter expression="false"
	 */
	private boolean validating;

	protected String[] getBindings() {
		return bindings;
	}

	protected List getClasspathElements() {
		return classpathElements;
	}

	protected String[] getDepends() {
		return depends;
	}

	protected boolean isExtension() {
		return extension;
	}


	protected boolean isForce() {
		return force;
	}

	protected String getPackageName() {
		return packageName;
	}

	protected String[] getProduces() {
		return produces;
	}

	protected MavenProject getProject() {
		return project;
	}

	protected Map getProperties() {
		return properties;
	}

	protected boolean isReadOnly() {
		return readOnly;
	}

	protected boolean isRemovingOldOutput() {
		return removingOldOutput;
	}

	protected String getSchemaReader() {
		return schemaReader;
	}

	protected String[] getSgFactoryChain() {
		return sgFactoryChain;
	}

	protected String getSrcTarget() {
		return srcTarget;
	}

	protected String getResourceTarget() {
		return resourceTarget;
	}

	protected boolean isValidating() {
		return validating;
	}

	private File[] getProducedFiles() {
		String[] prdcs = getProduces();
		if (prdcs == null) {
			String t1 = getSrcTarget();
			String t2 = getResourceTarget();
			if (t2 == null) {
				prdcs = new String[]{t1};
			} else {
				prdcs = new String[]{t1, t2};
			}
		}
		return getFiles(prdcs);
	}

	private File[] getFiles(String[] pSpec) {
		if (pSpec == null  ||  pSpec.length == 0) {
			return new File[0];
		}
		DirectoryScanner ds = new DirectoryScanner();
		final File baseDir = getProject().getBasedir();
		ds.setBasedir(baseDir);
		ds.setIncludes(pSpec);
		ds.scan();
		String[] files = ds.getIncludedFiles();
		File[] result = new File[files.length];
		for (int i = 0;  i < result.length;  i++) {
			result[i] = new File(baseDir, files[i]);
		}
		return result;
	}

	private File[] getSchemaFiles() {
		if (schemas == null  ||  schemas.length == 0) {
			schemas = new String[]{"src/jaxme/*.xsd"};
		}
		File[] schemaFiles = getFiles(schemas);
		if (schemaFiles.length == 0) {
			StringBuffer sb = new StringBuffer();
			sb.append("Schema specification returns no result: ");
			for (int i = 0;  i < schemas.length;  i++) {
				if (i > 0) {
					sb.append(",");
				}
				sb.append(schemas[i]);
			}
			getLog().warn(sb);
		}
		return schemaFiles;
	}

	private File[] getBindingFiles() {
		return getFiles(getBindings());
	}

	private File[] getDependencies() {
		return getFiles(getDepends());
	}

	private boolean isUptodate(File[] pSourceFiles, File[] pProducedFiles) {
		if (isForce()) {
			getLog().debug("Force flag set, disabling uptodate check.");
			return false;
		}
		if (pProducedFiles.length == 0) {
			getLog().debug("No produced files found, disabling uptodate check.");
			return false;
		}
		File minProducedFile = null;
		long minProducedTime = 0;
		for (int i = 0;  i < pProducedFiles.length;  i++) {
			File f = pProducedFiles[i];
			long l = f.lastModified();
			if (l == 0) {
				getLog().debug("Produced file " + f + " has unknown timestamp, disabling uptodate check.");
				return false;
			}
			if (minProducedTime == 0  ||  minProducedTime > l) {
				minProducedTime = l;
				minProducedFile = f;
			}
		}

		final File[] deps = pSourceFiles;
		long maxDepTime = 0;
		File maxDepFile = null;
		for (int i = 0;  i < deps.length;  i++) {
			File f = deps[i];
			long l = f.lastModified();
			if (l == 0) {
				getLog().debug("Dependency file " + f + " has unknown timestamp, disabling uptodate check.");
				return false;
			}
			if (maxDepTime == 0  ||  maxDepTime < l) {
				maxDepTime = l;
				maxDepFile = f;
			}
		}

		if (maxDepTime >= minProducedTime) {
			getLog().debug("Dependency file " + maxDepFile + " is more recent than produced file " + minProducedFile);
			return true;
		} else {
			getLog().debug("All produced files are uptodate.");
			return false;
		}
	}

	private File[] concat(File[] pFiles1, File[] pFiles2) {
		final File[] deps = new File[pFiles1.length + pFiles2.length];
		System.arraycopy(pFiles1, 0, deps, 0, pFiles1.length);
		System.arraycopy(pFiles2, 0, deps, pFiles1.length, pFiles2.length);
		return deps;
	}

	private void removeOldOutput(File[] pProducedFiles) throws MojoExecutionException {
		if (isRemovingOldOutput()) {
			for (int i = 0;  i < pProducedFiles.length;  i++) {
				File f = pProducedFiles[i];
				if (f.isFile()  &&  !f.delete()) {
					throw new MojoExecutionException("Unable to delete file: " + f.getAbsolutePath());
				}
			}
		}
	}

	private Class getSgFactoryChainClass(String pClass)
			throws MojoFailureException {
		Class c;
		try {
			c = Thread.currentThread().getContextClassLoader().loadClass(pClass);
		} catch (ClassNotFoundException e) {
			throw new MojoFailureException("The factory chain class " + pClass + " was not found.");
		}
		if (!SGFactoryChain.class.isAssignableFrom(c)) {
			throw new MojoFailureException("The factory chain class " + c.getName()
					+ " is not implementing " + SGFactoryChain.class);
		}
		return c;
	}

	private ClassLoader getClassLoader(ClassLoader pParent) throws MojoFailureException {
		List clElements = getClasspathElements();
		if (clElements == null  &&  clElements.size() == 0) {
			return pParent;
		}
		URL[] urls = new URL[clElements.size()];
		StringBuffer sb = new StringBuffer();
		for (int i = 0;  i < clElements.size();  i++) {
			final String elem = (String) clElements.get(i);
			File f = new File(elem);
			if (!f.isAbsolute()) {
				f = new File(getProject().getBasedir(), elem);
			}
			try {
				urls[i] = f.toURL();
			} catch (MalformedURLException e) {
				throw new MojoFailureException("Invalid classpath element: " + elem);
			}
			if (i > 0) {
				sb.append(File.pathSeparator);
			}
			sb.append(urls[i]);
		}
		getLog().debug("Using classpath " + sb);
		return new URLClassLoader(urls, pParent);
	}

	private SchemaReader getSchemaReaderInstance() throws MojoFailureException, MojoExecutionException {
		final SchemaReader result = newSchemaReaderInstance();
		getLog().debug("Schema reader class: " + result.getClass().getName());

		String[] chains = getSgFactoryChain();
		if (chains != null) {
			for (int i = 0;  i < chains.length;  i++) {
				Class c = getSgFactoryChainClass(chains[i]);
				getLog().debug("Adding SG Factory chain: " + c.getName());
				result.addSGFactoryChain(c);
			}
		}

		return result;
	}

	private SchemaReader newSchemaReaderInstance() throws MojoFailureException, MojoExecutionException {
		final SchemaReader result;
		final String s = getSchemaReader();
		if (s == null  ||  s.length() == 0) {
			if (isExtension()) {
				result = new JaxMeSchemaReader();
			} else {
				result = new JAXBSchemaReader();
			}
		} else {
			Class c;
			try {
				c = Thread.currentThread().getContextClassLoader().loadClass(s);
			} catch (ClassNotFoundException e) {
				throw new MojoFailureException("The schema reader class " + s + " was not found.");
			}
			Object o;
			try {
				o = c.newInstance();
			} catch (InstantiationException e) {
				throw new MojoExecutionException("Failed to instantiate schema reader class " + c.getName(), e);
			} catch (IllegalAccessException e) {
				throw new MojoExecutionException("Illegal access to schema reader class " + c.getName(), e);
			}
			try {
				result = (SchemaReader) o;
			} catch (ClassCastException e) {
				throw new MojoFailureException("The configured schema reader class " + c.getName()
						+ " is not implementing " + SchemaReader.class.getName());
			}
		}
		return result;
	}

	private File getSrcTargetDirectory() {
		return getTargetDir(getSrcTarget());
	}

	private File getTargetDir(String pDir) {
		File f = new File(pDir);
		if (!f.isAbsolute()) {
			f = new File(getProject().getBasedir(), getSrcTarget());
		}
		return f;
	}

	private File getResourceTargetDirectory() {
		return getTargetDir(getResourceTarget());
	}

	public void execute() throws MojoExecutionException, MojoFailureException {
		ClassLoader oldCl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(getClassLoader(SchemaReader.class.getClassLoader()));
		LoggerFactory lf = LoggerAccess.getLoggerFactory();
		try {
			LoggerAccess.setLoggerFactory(new LoggerFactory(){
				public Logger getLogger(String pName) {
					return new MavenProjectLogger(getLog(), pName);
				}
			});
			final File[] schemaFiles = getSchemaFiles();
			if (schemaFiles.length == 0) {
				return;
			}
			final File[] producedFiles = getProducedFiles();
			final File[] dependencies = getDependencies();
			final File[] bindingFiles = getBindingFiles();
			final boolean uptodate = isUptodate(concat(concat(schemaFiles, dependencies), bindingFiles), producedFiles);
			if (uptodate) {
				getLog().info("Generated files are uptodate.");
				return;
			}
	
			removeOldOutput(producedFiles);
	
			Generator g = new GeneratorImpl();
			for (int i = 0;  i < bindingFiles.length;  i++) {
				File f = bindingFiles[i];
				try {
					g.addBindings(new InputSource(f.toURL().toExternalForm()));
				} catch (ParserConfigurationException e) {
					throw new MojoExecutionException("Failed to add binding file "
							+ f.getPath() + ": " + e.getMessage(), e);
				} catch (SAXException e) {
					throw new MojoExecutionException("Failed to add binding file "
							+ f.getPath() + ": " + e.getMessage(), e);
				} catch (IOException e) {
					throw new MojoExecutionException("Failed to add binding file "
							+ f.getPath() + ": " + e.getMessage(), e);
				}
			}
			for (int i = 0;  i < schemaFiles.length;  i++) {
				final SchemaReader reader = getSchemaReaderInstance();
				g.setSchemaReader(reader);
				g.setForcingOverwrite(isForce());
				g.setSettingReadOnly(isReadOnly());
				g.setTargetDirectory(getSrcTargetDirectory());
				g.setResourceTargetDirectory(getResourceTargetDirectory());
				g.setValidating(isValidating());
				Map props = getProperties();
				if (props != null) {
					for (Iterator iter = props.entrySet().iterator();  iter.hasNext();  ) {
						Map.Entry entry = (Map.Entry) iter.next();
						g.setProperty((String) entry.getKey(), (String) entry.getValue());
					}
				}
				try {
					g.generate(schemaFiles[i]);
				} catch (Exception e) {
					throw new MojoExecutionException(e.getMessage(), e);
				}
			}
	
			getProject().addCompileSourceRoot(getSrcTarget());
			Resource resource = new Resource();
			resource.setDirectory(getResourceTarget());
			getProject().addResource(resource);
		} finally {
			Thread.currentThread().setContextClassLoader(oldCl);
			LoggerAccess.setLoggerFactory(lf);
		}
	}
}
