/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.analytics.util;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import org.apache.solr.analytics.request.RangeFacetRequest;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.schema.FieldType;
import org.apache.solr.schema.SchemaField;
import org.apache.solr.schema.TrieDateField;
import org.apache.solr.schema.TrieField;
import org.apache.solr.util.DateMathParser;

public abstract class RangeEndpointCalculator<T extends Comparable<T>> {
    protected final SchemaField field;
    protected final RangeFacetRequest request;

    public RangeEndpointCalculator(RangeFacetRequest request) {
        this.field = request.getField();
        this.request = request;
    }

    public String formatValue(T val) {
        return val.toString();
    }

    public final T getValue(String rawval) {
        try {
            return this.parseVal(rawval);
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Can't parse value " + rawval + " for field: " + this.field.getName(), (Throwable)e);
        }
    }

    protected abstract T parseVal(String var1) throws ParseException;

    public final Object getGap(String gap) {
        try {
            return this.parseGap(gap);
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Can't parse gap " + gap + " for field: " + this.field.getName(), (Throwable)e);
        }
    }

    protected Object parseGap(String rawval) throws ParseException {
        return this.parseVal(rawval);
    }

    public final T addGap(T value, String gap) {
        try {
            return this.parseAndAddGap(value, gap);
        }
        catch (Exception e) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Can't add gap " + gap + " to value " + value + " for field: " + this.field.getName(), (Throwable)e);
        }
    }

    protected abstract T parseAndAddGap(T var1, String var2) throws ParseException;

    public List<FacetRange> getRanges() {
        T start = this.getValue(this.request.getStart());
        T end = this.getValue(this.request.getEnd());
        if (end.compareTo(start) < 0) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "range facet 'end' comes before 'start': " + end + " < " + start);
        }
        String[] gaps = this.request.getGaps();
        String gap = gaps[0];
        EnumSet<FacetParams.FacetRangeInclude> include = this.request.getInclude();
        T low = start;
        ArrayList<FacetRange> ranges = new ArrayList<FacetRange>();
        int gapCounter = 0;
        while (low.compareTo(end) < 0) {
            T high;
            if (gapCounter < gaps.length) {
                gap = gaps[gapCounter++];
            }
            if (end.compareTo(high = this.addGap(low, gap)) < 0) {
                if (this.request.isHardEnd()) {
                    high = end;
                } else {
                    end = high;
                }
            }
            if (high.compareTo(low) < 0) {
                throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "range facet infinite loop (is gap negative? did the math overflow?)");
            }
            if (high.compareTo(low) == 0) {
                throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "range facet infinite loop: gap is either zero, or too small relative start/end and caused underflow: " + low + " + " + gap + " = " + high);
            }
            boolean includeLower = include.contains(FacetParams.FacetRangeInclude.ALL) || include.contains(FacetParams.FacetRangeInclude.LOWER) || include.contains(FacetParams.FacetRangeInclude.EDGE) && 0 == low.compareTo(start);
            boolean includeUpper = include.contains(FacetParams.FacetRangeInclude.ALL) || include.contains(FacetParams.FacetRangeInclude.UPPER) || include.contains(FacetParams.FacetRangeInclude.EDGE) && 0 == high.compareTo(end);
            String lowS = this.formatValue(low);
            String highS = this.formatValue(high);
            ranges.add(new FacetRange(lowS, lowS, highS, includeLower, includeUpper));
            low = high;
        }
        EnumSet<FacetParams.FacetRangeOther> others = this.request.getOthers();
        if (null != others && 0 < others.size() && !others.contains(FacetParams.FacetRangeOther.NONE)) {
            boolean all = others.contains(FacetParams.FacetRangeOther.ALL);
            if (all || others.contains(FacetParams.FacetRangeOther.BEFORE)) {
                ranges.add(new FacetRange(FacetParams.FacetRangeOther.BEFORE.toString(), null, this.formatValue(start), false, include.contains(FacetParams.FacetRangeInclude.OUTER) || include.contains(FacetParams.FacetRangeInclude.ALL) || !include.contains(FacetParams.FacetRangeInclude.LOWER) && !include.contains(FacetParams.FacetRangeInclude.EDGE)));
            }
            if (all || others.contains(FacetParams.FacetRangeOther.AFTER)) {
                ranges.add(new FacetRange(FacetParams.FacetRangeOther.AFTER.toString(), this.formatValue(end), null, include.contains(FacetParams.FacetRangeInclude.OUTER) || include.contains(FacetParams.FacetRangeInclude.ALL) || !include.contains(FacetParams.FacetRangeInclude.UPPER) && !include.contains(FacetParams.FacetRangeInclude.EDGE), false));
            }
            if (all || others.contains(FacetParams.FacetRangeOther.BETWEEN)) {
                ranges.add(new FacetRange(FacetParams.FacetRangeOther.BETWEEN.toString(), this.formatValue(start), this.formatValue(end), include.contains(FacetParams.FacetRangeInclude.LOWER) || include.contains(FacetParams.FacetRangeInclude.EDGE) || include.contains(FacetParams.FacetRangeInclude.ALL), include.contains(FacetParams.FacetRangeInclude.UPPER) || include.contains(FacetParams.FacetRangeInclude.EDGE) || include.contains(FacetParams.FacetRangeInclude.ALL)));
            }
        }
        return ranges;
    }

    public static RangeEndpointCalculator<? extends Comparable<?>> create(RangeFacetRequest request) {
        RangeEndpointCalculator calc;
        SchemaField sf = request.getField();
        FieldType ft = sf.getType();
        if (ft instanceof TrieField) {
            TrieField trie = (TrieField)ft;
            switch (trie.getType()) {
                case FLOAT: {
                    calc = new FloatRangeEndpointCalculator(request);
                    break;
                }
                case DOUBLE: {
                    calc = new DoubleRangeEndpointCalculator(request);
                    break;
                }
                case INTEGER: {
                    calc = new IntegerRangeEndpointCalculator(request);
                    break;
                }
                case LONG: {
                    calc = new LongRangeEndpointCalculator(request);
                    break;
                }
                case DATE: {
                    calc = new DateRangeEndpointCalculator(request, null);
                    break;
                }
                default: {
                    throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Unable to range facet on tried field of unexpected type:" + sf.getName());
                }
            }
        } else {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Unable to range facet on field:" + sf);
        }
        return calc;
    }

    public static class DateRangeEndpointCalculator
    extends RangeEndpointCalculator<Date> {
        private final Date now;

        public DateRangeEndpointCalculator(RangeFacetRequest request, Date now) {
            super(request);
            this.now = now;
            if (!(this.field.getType() instanceof TrieDateField)) {
                throw new IllegalArgumentException("SchemaField must use field type extending TrieDateField");
            }
        }

        @Override
        public String formatValue(Date val) {
            return val.toInstant().toString();
        }

        @Override
        protected Date parseVal(String rawval) {
            return DateMathParser.parseMath((Date)this.now, (String)rawval);
        }

        @Override
        protected Object parseGap(String rawval) {
            return rawval;
        }

        @Override
        public Date parseAndAddGap(Date value, String gap) throws ParseException {
            DateMathParser dmp = new DateMathParser();
            dmp.setNow(value);
            return dmp.parseMath(gap);
        }
    }

    public static class LongRangeEndpointCalculator
    extends RangeEndpointCalculator<Long> {
        public LongRangeEndpointCalculator(RangeFacetRequest request) {
            super(request);
        }

        @Override
        protected Long parseVal(String rawval) {
            return Long.valueOf(rawval);
        }

        @Override
        public Long parseAndAddGap(Long value, String gap) {
            return new Long(value + Long.valueOf(gap));
        }
    }

    public static class IntegerRangeEndpointCalculator
    extends RangeEndpointCalculator<Integer> {
        public IntegerRangeEndpointCalculator(RangeFacetRequest request) {
            super(request);
        }

        @Override
        protected Integer parseVal(String rawval) {
            return Integer.valueOf(rawval);
        }

        @Override
        public Integer parseAndAddGap(Integer value, String gap) {
            return new Integer(value + Integer.valueOf(gap));
        }
    }

    public static class DoubleRangeEndpointCalculator
    extends RangeEndpointCalculator<Double> {
        public DoubleRangeEndpointCalculator(RangeFacetRequest request) {
            super(request);
        }

        @Override
        protected Double parseVal(String rawval) {
            return Double.valueOf(rawval);
        }

        @Override
        public Double parseAndAddGap(Double value, String gap) {
            return new Double(value + Double.valueOf(gap));
        }
    }

    public static class FloatRangeEndpointCalculator
    extends RangeEndpointCalculator<Float> {
        public FloatRangeEndpointCalculator(RangeFacetRequest request) {
            super(request);
        }

        @Override
        protected Float parseVal(String rawval) {
            return Float.valueOf(rawval);
        }

        @Override
        public Float parseAndAddGap(Float value, String gap) {
            return new Float(value.floatValue() + Float.valueOf(gap).floatValue());
        }
    }

    public static class FacetRange {
        public final String name;
        public final String lower;
        public final String upper;
        public final boolean includeLower;
        public final boolean includeUpper;

        public FacetRange(String name, String lower, String upper, boolean includeLower, boolean includeUpper) {
            this.name = name;
            this.lower = lower;
            this.upper = upper;
            this.includeLower = includeLower;
            this.includeUpper = includeUpper;
        }
    }
}

