/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.json;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.TimeZone;
import org.apache.kafka.common.cache.Cache;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.Decimal;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaAndValue;
import org.apache.kafka.connect.data.SchemaBuilder;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.data.Time;
import org.apache.kafka.connect.data.Timestamp;
import org.apache.kafka.connect.errors.DataException;
import org.apache.kafka.connect.json.JsonConverter;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.powermock.reflect.Whitebox;

public class JsonConverterTest {
    private static final String TOPIC = "topic";
    ObjectMapper objectMapper = new ObjectMapper();
    JsonConverter converter = new JsonConverter();

    @Before
    public void setUp() {
        this.converter.configure(Collections.EMPTY_MAP, false);
    }

    @Test
    public void testConnectSchemaMetadataTranslation() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.BOOLEAN_SCHEMA, (Object)true), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\" }, \"payload\": true }".getBytes()));
        Assert.assertEquals((Object)new SchemaAndValue(Schema.OPTIONAL_BOOLEAN_SCHEMA, null), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\", \"optional\": true }, \"payload\": null }".getBytes()));
        Assert.assertEquals((Object)new SchemaAndValue(SchemaBuilder.bool().defaultValue((Object)true).build(), (Object)true), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\", \"default\": true }, \"payload\": null }".getBytes()));
        Assert.assertEquals((Object)new SchemaAndValue(SchemaBuilder.bool().required().name("bool").version(Integer.valueOf(2)).doc("the documentation").parameter("foo", "bar").build(), (Object)true), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\", \"optional\": false, \"name\": \"bool\", \"version\": 2, \"doc\": \"the documentation\", \"parameters\": { \"foo\": \"bar\" }}, \"payload\": true }".getBytes()));
    }

    @Test
    public void booleanToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.BOOLEAN_SCHEMA, (Object)true), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\" }, \"payload\": true }".getBytes()));
        Assert.assertEquals((Object)new SchemaAndValue(Schema.BOOLEAN_SCHEMA, (Object)false), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\" }, \"payload\": false }".getBytes()));
    }

    @Test
    public void byteToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.INT8_SCHEMA, (Object)12), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"int8\" }, \"payload\": 12 }".getBytes()));
    }

    @Test
    public void shortToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.INT16_SCHEMA, (Object)12), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"int16\" }, \"payload\": 12 }".getBytes()));
    }

    @Test
    public void intToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.INT32_SCHEMA, (Object)12), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"int32\" }, \"payload\": 12 }".getBytes()));
    }

    @Test
    public void longToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.INT64_SCHEMA, (Object)12L), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"int64\" }, \"payload\": 12 }".getBytes()));
        Assert.assertEquals((Object)new SchemaAndValue(Schema.INT64_SCHEMA, (Object)0x40000000000L), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"int64\" }, \"payload\": 4398046511104 }".getBytes()));
    }

    @Test
    public void floatToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.FLOAT32_SCHEMA, (Object)Float.valueOf(12.34f)), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"float\" }, \"payload\": 12.34 }".getBytes()));
    }

    @Test
    public void doubleToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.FLOAT64_SCHEMA, (Object)12.34), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"double\" }, \"payload\": 12.34 }".getBytes()));
    }

    @Test
    public void bytesToConnect() throws UnsupportedEncodingException {
        ByteBuffer reference = ByteBuffer.wrap("test-string".getBytes("UTF-8"));
        String msg = "{ \"schema\": { \"type\": \"bytes\" }, \"payload\": \"dGVzdC1zdHJpbmc=\" }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        ByteBuffer converted = ByteBuffer.wrap((byte[])schemaAndValue.value());
        Assert.assertEquals((Object)reference, (Object)converted);
    }

    @Test
    public void stringToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.STRING_SCHEMA, (Object)"foo-bar-baz"), (Object)this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"string\" }, \"payload\": \"foo-bar-baz\" }".getBytes()));
    }

    @Test
    public void arrayToConnect() {
        byte[] arrayJson = "{ \"schema\": { \"type\": \"array\", \"items\": { \"type\" : \"int32\" } }, \"payload\": [1, 2, 3] }".getBytes();
        Assert.assertEquals((Object)new SchemaAndValue(SchemaBuilder.array((Schema)Schema.INT32_SCHEMA).build(), Arrays.asList(1, 2, 3)), (Object)this.converter.toConnectData(TOPIC, arrayJson));
    }

    @Test
    public void mapToConnectStringKeys() {
        byte[] mapJson = "{ \"schema\": { \"type\": \"map\", \"keys\": { \"type\" : \"string\" }, \"values\": { \"type\" : \"int32\" } }, \"payload\": { \"key1\": 12, \"key2\": 15} }".getBytes();
        HashMap<String, Integer> expected = new HashMap<String, Integer>();
        expected.put("key1", 12);
        expected.put("key2", 15);
        Assert.assertEquals((Object)new SchemaAndValue(SchemaBuilder.map((Schema)Schema.STRING_SCHEMA, (Schema)Schema.INT32_SCHEMA).build(), expected), (Object)this.converter.toConnectData(TOPIC, mapJson));
    }

    @Test
    public void mapToConnectNonStringKeys() {
        byte[] mapJson = "{ \"schema\": { \"type\": \"map\", \"keys\": { \"type\" : \"int32\" }, \"values\": { \"type\" : \"int32\" } }, \"payload\": [ [1, 12], [2, 15] ] }".getBytes();
        HashMap<Integer, Integer> expected = new HashMap<Integer, Integer>();
        expected.put(1, 12);
        expected.put(2, 15);
        Assert.assertEquals((Object)new SchemaAndValue(SchemaBuilder.map((Schema)Schema.INT32_SCHEMA, (Schema)Schema.INT32_SCHEMA).build(), expected), (Object)this.converter.toConnectData(TOPIC, mapJson));
    }

    @Test
    public void structToConnect() {
        byte[] structJson = "{ \"schema\": { \"type\": \"struct\", \"fields\": [{ \"field\": \"field1\", \"type\": \"boolean\" }, { \"field\": \"field2\", \"type\": \"string\" }] }, \"payload\": { \"field1\": true, \"field2\": \"string\" } }".getBytes();
        Schema expectedSchema = SchemaBuilder.struct().field("field1", Schema.BOOLEAN_SCHEMA).field("field2", Schema.STRING_SCHEMA).build();
        Struct expected = new Struct(expectedSchema).put("field1", (Object)true).put("field2", (Object)"string");
        SchemaAndValue converted = this.converter.toConnectData(TOPIC, structJson);
        Assert.assertEquals((Object)new SchemaAndValue(expectedSchema, (Object)expected), (Object)converted);
    }

    @Test(expected=DataException.class)
    public void nullToConnect() {
        Assert.assertEquals((Object)SchemaAndValue.NULL, (Object)this.converter.toConnectData(TOPIC, null));
    }

    @Test
    public void nullSchemaPrimitiveToConnect() {
        SchemaAndValue converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": null }".getBytes());
        Assert.assertEquals((Object)SchemaAndValue.NULL, (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": true }".getBytes());
        Assert.assertEquals((Object)new SchemaAndValue(null, (Object)true), (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": 12 }".getBytes());
        Assert.assertEquals((Object)new SchemaAndValue(null, (Object)12L), (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": 12.24 }".getBytes());
        Assert.assertEquals((Object)new SchemaAndValue(null, (Object)12.24), (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": \"a string\" }".getBytes());
        Assert.assertEquals((Object)new SchemaAndValue(null, (Object)"a string"), (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": [1, \"2\", 3] }".getBytes());
        Assert.assertEquals((Object)new SchemaAndValue(null, Arrays.asList(1L, "2", 3L)), (Object)converted);
        converted = this.converter.toConnectData(TOPIC, "{ \"schema\": null, \"payload\": { \"field1\": 1, \"field2\": 2} }".getBytes());
        HashMap<String, Long> obj = new HashMap<String, Long>();
        obj.put("field1", 1L);
        obj.put("field2", 2L);
        Assert.assertEquals((Object)new SchemaAndValue(null, obj), (Object)converted);
    }

    @Test
    public void decimalToConnect() {
        Schema schema = Decimal.schema((int)2);
        BigDecimal reference = new BigDecimal(new BigInteger("156"), 2);
        String msg = "{ \"schema\": { \"type\": \"bytes\", \"name\": \"org.apache.kafka.connect.data.Decimal\", \"version\": 1, \"parameters\": { \"scale\": \"2\" } }, \"payload\": \"AJw=\" }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        BigDecimal converted = (BigDecimal)schemaAndValue.value();
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)converted);
    }

    @Test
    public void decimalToConnectOptional() {
        Schema schema = Decimal.builder((int)2).optional().schema();
        String msg = "{ \"schema\": { \"type\": \"bytes\", \"name\": \"org.apache.kafka.connect.data.Decimal\", \"version\": 1, \"optional\": true, \"parameters\": { \"scale\": \"2\" } }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertNull((Object)schemaAndValue.value());
    }

    @Test
    public void decimalToConnectWithDefaultValue() {
        BigDecimal reference = new BigDecimal(new BigInteger("156"), 2);
        Schema schema = Decimal.builder((int)2).defaultValue((Object)reference).build();
        String msg = "{ \"schema\": { \"type\": \"bytes\", \"name\": \"org.apache.kafka.connect.data.Decimal\", \"version\": 1, \"default\": \"AJw=\", \"parameters\": { \"scale\": \"2\" } }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void decimalToConnectOptionalWithDefaultValue() {
        BigDecimal reference = new BigDecimal(new BigInteger("156"), 2);
        Schema schema = Decimal.builder((int)2).optional().defaultValue((Object)reference).build();
        String msg = "{ \"schema\": { \"type\": \"bytes\", \"name\": \"org.apache.kafka.connect.data.Decimal\", \"version\": 1, \"optional\": true, \"default\": \"AJw=\", \"parameters\": { \"scale\": \"2\" } }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void dateToConnect() {
        Schema schema = org.apache.kafka.connect.data.Date.SCHEMA;
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(5, 10000);
        Date reference = calendar.getTime();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Date\", \"version\": 1 }, \"payload\": 10000 }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Date converted = (Date)schemaAndValue.value();
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)converted);
    }

    @Test
    public void dateToConnectOptional() {
        Schema schema = org.apache.kafka.connect.data.Date.builder().optional().schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Date\", \"version\": 1, \"optional\": true }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertNull((Object)schemaAndValue.value());
    }

    @Test
    public void dateToConnectWithDefaultValue() {
        Date reference = new Date(0L);
        Schema schema = org.apache.kafka.connect.data.Date.builder().defaultValue((Object)reference).schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Date\", \"version\": 1, \"default\": 0 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void dateToConnectOptionalWithDefaultValue() {
        Date reference = new Date(0L);
        Schema schema = org.apache.kafka.connect.data.Date.builder().optional().defaultValue((Object)reference).schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Date\", \"version\": 1, \"optional\": true, \"default\": 0 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void timeToConnect() {
        Schema schema = Time.SCHEMA;
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(14, 14400000);
        Date reference = calendar.getTime();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Time\", \"version\": 1 }, \"payload\": 14400000 }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Date converted = (Date)schemaAndValue.value();
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)converted);
    }

    @Test
    public void timeToConnectOptional() {
        Schema schema = Time.builder().optional().schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Time\", \"version\": 1, \"optional\": true }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertNull((Object)schemaAndValue.value());
    }

    @Test
    public void timeToConnectWithDefaultValue() {
        Date reference = new Date(0L);
        Schema schema = Time.builder().defaultValue((Object)reference).schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Time\", \"version\": 1, \"default\": 0 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void timeToConnectOptionalWithDefaultValue() {
        Date reference = new Date(0L);
        Schema schema = Time.builder().optional().defaultValue((Object)reference).schema();
        String msg = "{ \"schema\": { \"type\": \"int32\", \"name\": \"org.apache.kafka.connect.data.Time\", \"version\": 1, \"optional\": true, \"default\": 0 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)schemaAndValue.value());
    }

    @Test
    public void timestampToConnect() {
        Schema schema = Timestamp.SCHEMA;
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(14, 2000000000);
        calendar.add(14, 2000000000);
        Date reference = calendar.getTime();
        String msg = "{ \"schema\": { \"type\": \"int64\", \"name\": \"org.apache.kafka.connect.data.Timestamp\", \"version\": 1 }, \"payload\": 4000000000 }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Date converted = (Date)schemaAndValue.value();
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)reference, (Object)converted);
    }

    @Test
    public void timestampToConnectOptional() {
        Schema schema = Timestamp.builder().optional().schema();
        String msg = "{ \"schema\": { \"type\": \"int64\", \"name\": \"org.apache.kafka.connect.data.Timestamp\", \"version\": 1, \"optional\": true }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertNull((Object)schemaAndValue.value());
    }

    @Test
    public void timestampToConnectWithDefaultValue() {
        Schema schema = Timestamp.builder().defaultValue((Object)new Date(42L)).schema();
        String msg = "{ \"schema\": { \"type\": \"int64\", \"name\": \"org.apache.kafka.connect.data.Timestamp\", \"version\": 1, \"default\": 42 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)new Date(42L), (Object)schemaAndValue.value());
    }

    @Test
    public void timestampToConnectOptionalWithDefaultValue() {
        Schema schema = Timestamp.builder().optional().defaultValue((Object)new Date(42L)).schema();
        String msg = "{ \"schema\": { \"type\": \"int64\", \"name\": \"org.apache.kafka.connect.data.Timestamp\", \"version\": 1,  \"optional\": true, \"default\": 42 }, \"payload\": null }";
        SchemaAndValue schemaAndValue = this.converter.toConnectData(TOPIC, msg.getBytes());
        Assert.assertEquals((Object)schema, (Object)schemaAndValue.schema());
        Assert.assertEquals((Object)new Date(42L), (Object)schemaAndValue.value());
    }

    @Test
    public void testJsonSchemaMetadataTranslation() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.BOOLEAN_SCHEMA, (Object)true));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"boolean\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)true, (Object)converted.get("payload").booleanValue());
        converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.OPTIONAL_BOOLEAN_SCHEMA, null));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"boolean\", \"optional\": true }"), (Object)converted.get("schema"));
        Assert.assertTrue((boolean)converted.get("payload").isNull());
        converted = this.parse(this.converter.fromConnectData(TOPIC, SchemaBuilder.bool().defaultValue((Object)true).build(), (Object)true));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"boolean\", \"optional\": false, \"default\": true }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)true, (Object)converted.get("payload").booleanValue());
        converted = this.parse(this.converter.fromConnectData(TOPIC, SchemaBuilder.bool().required().name("bool").version(Integer.valueOf(3)).doc("the documentation").parameter("foo", "bar").build(), (Object)true));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"boolean\", \"optional\": false, \"name\": \"bool\", \"version\": 3, \"doc\": \"the documentation\", \"parameters\": { \"foo\": \"bar\" }}"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)true, (Object)converted.get("payload").booleanValue());
    }

    @Test
    public void testCacheSchemaToConnectConversion() {
        Cache cache = (Cache)Whitebox.getInternalState((Object)this.converter, (String)"toConnectSchemaCache");
        Assert.assertEquals((long)0L, (long)cache.size());
        this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\" }, \"payload\": true }".getBytes());
        Assert.assertEquals((long)1L, (long)cache.size());
        this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\" }, \"payload\": true }".getBytes());
        Assert.assertEquals((long)1L, (long)cache.size());
        this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\", \"optional\": true }, \"payload\": true }".getBytes());
        Assert.assertEquals((long)2L, (long)cache.size());
        this.converter.toConnectData(TOPIC, "{ \"schema\": { \"type\": \"boolean\", \"optional\": false }, \"payload\": true }".getBytes());
        Assert.assertEquals((long)3L, (long)cache.size());
    }

    @Test
    public void booleanToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.BOOLEAN_SCHEMA, (Object)true));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"boolean\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)true, (Object)converted.get("payload").booleanValue());
    }

    @Test
    public void byteToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.INT8_SCHEMA, (Object)12));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int8\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((long)12L, (long)converted.get("payload").intValue());
    }

    @Test
    public void shortToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.INT16_SCHEMA, (Object)12));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int16\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((long)12L, (long)converted.get("payload").intValue());
    }

    @Test
    public void intToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.INT32_SCHEMA, (Object)12));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int32\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((long)12L, (long)converted.get("payload").intValue());
    }

    @Test
    public void longToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.INT64_SCHEMA, (Object)0x40000000000L));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int64\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((long)0x40000000000L, (long)converted.get("payload").longValue());
    }

    @Test
    public void floatToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.FLOAT32_SCHEMA, (Object)Float.valueOf(12.34f)));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"float\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((double)12.34f, (double)converted.get("payload").floatValue(), (double)0.001);
    }

    @Test
    public void doubleToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.FLOAT64_SCHEMA, (Object)12.34));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"double\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((double)12.34, (double)converted.get("payload").doubleValue(), (double)0.001);
    }

    @Test
    public void bytesToJson() throws IOException {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.BYTES_SCHEMA, (Object)"test-string".getBytes()));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"bytes\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)ByteBuffer.wrap("test-string".getBytes()), (Object)ByteBuffer.wrap(converted.get("payload").binaryValue()));
    }

    @Test
    public void stringToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Schema.STRING_SCHEMA, (Object)"test-string"));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"string\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)"test-string", (Object)converted.get("payload").textValue());
    }

    @Test
    public void arrayToJson() {
        Schema int32Array = SchemaBuilder.array((Schema)Schema.INT32_SCHEMA).build();
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, int32Array, Arrays.asList(1, 2, 3)));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"array\", \"items\": { \"type\": \"int32\", \"optional\": false }, \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)JsonNodeFactory.instance.arrayNode().add(1).add(2).add(3), (Object)converted.get("payload"));
    }

    @Test
    public void mapToJsonStringKeys() {
        Schema stringIntMap = SchemaBuilder.map((Schema)Schema.STRING_SCHEMA, (Schema)Schema.INT32_SCHEMA).build();
        HashMap<String, Integer> input = new HashMap<String, Integer>();
        input.put("key1", 12);
        input.put("key2", 15);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, stringIntMap, input));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"map\", \"keys\": { \"type\" : \"string\", \"optional\": false }, \"values\": { \"type\" : \"int32\", \"optional\": false }, \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)JsonNodeFactory.instance.objectNode().put("key1", 12).put("key2", 15), (Object)converted.get("payload"));
    }

    @Test
    public void mapToJsonNonStringKeys() {
        Schema intIntMap = SchemaBuilder.map((Schema)Schema.INT32_SCHEMA, (Schema)Schema.INT32_SCHEMA).build();
        HashMap<Integer, Integer> input = new HashMap<Integer, Integer>();
        input.put(1, 12);
        input.put(2, 15);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, intIntMap, input));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"map\", \"keys\": { \"type\" : \"int32\", \"optional\": false }, \"values\": { \"type\" : \"int32\", \"optional\": false }, \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertTrue((boolean)converted.get("payload").isArray());
        ArrayNode payload = (ArrayNode)converted.get("payload");
        Assert.assertEquals((long)2L, (long)payload.size());
        HashSet<JsonNode> payloadEntries = new HashSet<JsonNode>();
        for (JsonNode elem : payload) {
            payloadEntries.add(elem);
        }
        Assert.assertEquals(new HashSet<ArrayNode>(Arrays.asList(JsonNodeFactory.instance.arrayNode().add(1).add(12), JsonNodeFactory.instance.arrayNode().add(2).add(15))), payloadEntries);
    }

    @Test
    public void structToJson() {
        Schema schema = SchemaBuilder.struct().field("field1", Schema.BOOLEAN_SCHEMA).field("field2", Schema.STRING_SCHEMA).field("field3", Schema.STRING_SCHEMA).field("field4", Schema.BOOLEAN_SCHEMA).build();
        Struct input = new Struct(schema).put("field1", (Object)true).put("field2", (Object)"string2").put("field3", (Object)"string3").put("field4", (Object)false);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, schema, (Object)input));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"struct\", \"optional\": false, \"fields\": [{ \"field\": \"field1\", \"type\": \"boolean\", \"optional\": false }, { \"field\": \"field2\", \"type\": \"string\", \"optional\": false }, { \"field\": \"field3\", \"type\": \"string\", \"optional\": false }, { \"field\": \"field4\", \"type\": \"boolean\", \"optional\": false }] }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)JsonNodeFactory.instance.objectNode().put("field1", true).put("field2", "string2").put("field3", "string3").put("field4", false), (Object)converted.get("payload"));
    }

    @Test
    public void structSchemaIdentical() {
        Schema schema = SchemaBuilder.struct().field("field1", Schema.BOOLEAN_SCHEMA).field("field2", Schema.STRING_SCHEMA).field("field3", Schema.STRING_SCHEMA).field("field4", Schema.BOOLEAN_SCHEMA).build();
        Schema inputSchema = SchemaBuilder.struct().field("field1", Schema.BOOLEAN_SCHEMA).field("field2", Schema.STRING_SCHEMA).field("field3", Schema.STRING_SCHEMA).field("field4", Schema.BOOLEAN_SCHEMA).build();
        Struct input = new Struct(inputSchema).put("field1", (Object)true).put("field2", (Object)"string2").put("field3", (Object)"string3").put("field4", (Object)false);
        this.assertStructSchemaEqual(schema, input);
    }

    @Test
    public void decimalToJson() throws IOException {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Decimal.schema((int)2), (Object)new BigDecimal(new BigInteger("156"), 2)));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"bytes\", \"optional\": false, \"name\": \"org.apache.kafka.connect.data.Decimal\", \"version\": 1, \"parameters\": { \"scale\": \"2\" } }"), (Object)converted.get("schema"));
        Assert.assertArrayEquals((byte[])new byte[]{0, -100}, (byte[])converted.get("payload").binaryValue());
    }

    @Test
    public void dateToJson() throws IOException {
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(5, 10000);
        Date date = calendar.getTime();
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, org.apache.kafka.connect.data.Date.SCHEMA, (Object)date));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int32\", \"optional\": false, \"name\": \"org.apache.kafka.connect.data.Date\", \"version\": 1 }"), (Object)converted.get("schema"));
        JsonNode payload = converted.get("payload");
        Assert.assertTrue((boolean)payload.isInt());
        Assert.assertEquals((long)10000L, (long)payload.intValue());
    }

    @Test
    public void timeToJson() throws IOException {
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(14, 14400000);
        Date date = calendar.getTime();
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Time.SCHEMA, (Object)date));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int32\", \"optional\": false, \"name\": \"org.apache.kafka.connect.data.Time\", \"version\": 1 }"), (Object)converted.get("schema"));
        JsonNode payload = converted.get("payload");
        Assert.assertTrue((boolean)payload.isInt());
        Assert.assertEquals((long)14400000L, (long)payload.longValue());
    }

    @Test
    public void timestampToJson() throws IOException {
        GregorianCalendar calendar = new GregorianCalendar(1970, 0, 1, 0, 0, 0);
        calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
        calendar.add(14, 2000000000);
        calendar.add(14, 2000000000);
        Date date = calendar.getTime();
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, Timestamp.SCHEMA, (Object)date));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"int64\", \"optional\": false, \"name\": \"org.apache.kafka.connect.data.Timestamp\", \"version\": 1 }"), (Object)converted.get("schema"));
        JsonNode payload = converted.get("payload");
        Assert.assertTrue((boolean)payload.isLong());
        Assert.assertEquals((long)4000000000L, (long)payload.longValue());
    }

    @Test
    public void nullSchemaAndPrimitiveToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, null, (Object)true));
        this.validateEnvelopeNullSchema(converted);
        Assert.assertTrue((boolean)converted.get("schema").isNull());
        Assert.assertEquals((Object)true, (Object)converted.get("payload").booleanValue());
    }

    @Test
    public void nullSchemaAndArrayToJson() {
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, null, Arrays.asList(1, "string", true)));
        this.validateEnvelopeNullSchema(converted);
        Assert.assertTrue((boolean)converted.get("schema").isNull());
        Assert.assertEquals((Object)JsonNodeFactory.instance.arrayNode().add(1).add("string").add(true), (Object)converted.get("payload"));
    }

    @Test
    public void nullSchemaAndMapToJson() {
        HashMap<String, Object> input = new HashMap<String, Object>();
        input.put("key1", 12);
        input.put("key2", "string");
        input.put("key3", true);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, null, input));
        this.validateEnvelopeNullSchema(converted);
        Assert.assertTrue((boolean)converted.get("schema").isNull());
        Assert.assertEquals((Object)JsonNodeFactory.instance.objectNode().put("key1", 12).put("key2", "string").put("key3", true), (Object)converted.get("payload"));
    }

    @Test
    public void nullSchemaAndMapNonStringKeysToJson() {
        HashMap<Object, Object> input = new HashMap<Object, Object>();
        input.put("string", 12);
        input.put(52, "string");
        input.put(false, true);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, null, input));
        this.validateEnvelopeNullSchema(converted);
        Assert.assertTrue((boolean)converted.get("schema").isNull());
        Assert.assertTrue((boolean)converted.get("payload").isArray());
        ArrayNode payload = (ArrayNode)converted.get("payload");
        Assert.assertEquals((long)3L, (long)payload.size());
        HashSet<JsonNode> payloadEntries = new HashSet<JsonNode>();
        for (JsonNode elem : payload) {
            payloadEntries.add(elem);
        }
        Assert.assertEquals(new HashSet<ArrayNode>(Arrays.asList(JsonNodeFactory.instance.arrayNode().add("string").add(12), JsonNodeFactory.instance.arrayNode().add(52).add("string"), JsonNodeFactory.instance.arrayNode().add(false).add(true))), payloadEntries);
    }

    @Test(expected=DataException.class)
    public void mismatchSchemaJson() {
        this.converter.fromConnectData(TOPIC, Schema.FLOAT64_SCHEMA, (Object)true);
    }

    @Test
    public void noSchemaToConnect() {
        Map<String, Boolean> props = Collections.singletonMap("schemas.enable", false);
        this.converter.configure(props, true);
        Assert.assertEquals((Object)new SchemaAndValue(null, (Object)true), (Object)this.converter.toConnectData(TOPIC, "true".getBytes()));
    }

    @Test
    public void noSchemaToJson() {
        Map<String, Boolean> props = Collections.singletonMap("schemas.enable", false);
        this.converter.configure(props, true);
        JsonNode converted = this.parse(this.converter.fromConnectData(TOPIC, null, (Object)true));
        Assert.assertTrue((boolean)converted.isBoolean());
        Assert.assertEquals((Object)true, (Object)converted.booleanValue());
    }

    @Test
    public void testCacheSchemaToJsonConversion() {
        Cache cache = (Cache)Whitebox.getInternalState((Object)this.converter, (String)"fromConnectSchemaCache");
        Assert.assertEquals((long)0L, (long)cache.size());
        this.converter.fromConnectData(TOPIC, SchemaBuilder.bool().build(), (Object)true);
        Assert.assertEquals((long)1L, (long)cache.size());
        this.converter.fromConnectData(TOPIC, SchemaBuilder.bool().build(), (Object)true);
        Assert.assertEquals((long)1L, (long)cache.size());
        this.converter.fromConnectData(TOPIC, SchemaBuilder.bool().optional().build(), (Object)true);
        Assert.assertEquals((long)2L, (long)cache.size());
    }

    @Test
    public void testJsonSchemaCacheSizeFromConfigFile() throws URISyntaxException, IOException {
        URL url = this.getClass().getResource("/connect-test.properties");
        File propFile = new File(url.toURI());
        String workerPropsFile = propFile.getAbsolutePath();
        Map workerProps = !workerPropsFile.isEmpty() ? Utils.propsToStringMap((Properties)Utils.loadProps((String)workerPropsFile)) : Collections.emptyMap();
        JsonConverter rc = new JsonConverter();
        rc.configure(workerProps, false);
    }

    @Test
    public void testStringHeaderToJson() throws UnsupportedEncodingException {
        JsonNode converted = this.parse(this.converter.fromConnectHeader(TOPIC, "headerName", Schema.STRING_SCHEMA, (Object)"test-string"));
        this.validateEnvelope(converted);
        Assert.assertEquals((Object)this.parse("{ \"type\": \"string\", \"optional\": false }"), (Object)converted.get("schema"));
        Assert.assertEquals((Object)"test-string", (Object)converted.get("payload").textValue());
    }

    @Test
    public void stringHeaderToConnect() {
        Assert.assertEquals((Object)new SchemaAndValue(Schema.STRING_SCHEMA, (Object)"foo-bar-baz"), (Object)this.converter.toConnectHeader(TOPIC, "headerName", "{ \"schema\": { \"type\": \"string\" }, \"payload\": \"foo-bar-baz\" }".getBytes()));
    }

    private JsonNode parse(byte[] json) {
        try {
            return this.objectMapper.readTree(json);
        }
        catch (IOException e) {
            Assert.fail((String)("IOException during JSON parse: " + e.getMessage()));
            throw new RuntimeException("failed");
        }
    }

    private JsonNode parse(String json) {
        try {
            return this.objectMapper.readTree(json);
        }
        catch (IOException e) {
            Assert.fail((String)("IOException during JSON parse: " + e.getMessage()));
            throw new RuntimeException("failed");
        }
    }

    private void validateEnvelope(JsonNode env) {
        Assert.assertNotNull((Object)env);
        Assert.assertTrue((boolean)env.isObject());
        Assert.assertEquals((long)2L, (long)env.size());
        Assert.assertTrue((boolean)env.has("schema"));
        Assert.assertTrue((boolean)env.get("schema").isObject());
        Assert.assertTrue((boolean)env.has("payload"));
    }

    private void validateEnvelopeNullSchema(JsonNode env) {
        Assert.assertNotNull((Object)env);
        Assert.assertTrue((boolean)env.isObject());
        Assert.assertEquals((long)2L, (long)env.size());
        Assert.assertTrue((boolean)env.has("schema"));
        Assert.assertTrue((boolean)env.get("schema").isNull());
        Assert.assertTrue((boolean)env.has("payload"));
    }

    private void assertStructSchemaEqual(Schema schema, Struct struct) {
        this.converter.fromConnectData(TOPIC, schema, (Object)struct);
        Assert.assertEquals((Object)schema, (Object)struct.schema());
    }
}

