/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.johnzon.jaxrs;

import javax.json.Json;
import javax.json.JsonStructure;
import javax.json.JsonWriter;
import javax.json.JsonWriterFactory;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import java.io.Flushable;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collections;

@Provider
@Produces({
    "application/json", "*/json",
    "*/*+json", "*/x-json",
    "*/javascript", "*/x-javascript"
})
public class JsrMessageBodyWriter implements MessageBodyWriter<JsonStructure> {
    private final JsonWriterFactory factory;
    private final boolean close;

    public JsrMessageBodyWriter() {
        this(Json.createWriterFactory(Collections.<String, Object>emptyMap()), false);
    }

    public JsrMessageBodyWriter(final JsonWriterFactory factory, final boolean closeStreams) {
        this.factory = factory;
        this.close = closeStreams;
    }

    @Override
    public boolean isWriteable(final Class<?> aClass, final Type type,
                               final Annotation[] annotations, final MediaType mediaType) {
        return JsonStructure.class.isAssignableFrom(aClass);
    }

    @Override
    public long getSize(final JsonStructure jsonStructure, final Class<?> aClass,
                        final Type type, final Annotation[] annotations,
                        final MediaType mediaType) {
        return -1;
    }

    @Override
    public void writeTo(final JsonStructure jsonStructure,
                        final Class<?> aClass, final Type type,
                        final Annotation[] annotations, final MediaType mediaType,
                        final MultivaluedMap<String, Object> stringObjectMultivaluedMap,
                        final OutputStream outputStream) throws IOException, WebApplicationException {
        JsonWriter writer = null;
        try {
            writer = factory.createWriter(outputStream);
            writer.write(jsonStructure);
        } finally {
            if (writer != null) {
                if (close) {
                    writer.close();
                } else if (Flushable.class.isInstance(writer)) {
                    Flushable.class.cast(writer).flush();
                }
            }
        }
    }
}
