/*
 * Decompiled with CFR 0.152.
 */
package brooklyn.policy.autoscaling;

import brooklyn.catalog.Catalog;
import brooklyn.config.ConfigKey;
import brooklyn.entity.Entity;
import brooklyn.entity.basic.Entities;
import brooklyn.entity.basic.EntityLocal;
import brooklyn.entity.trait.Resizable;
import brooklyn.entity.trait.Startable;
import brooklyn.event.AttributeSensor;
import brooklyn.event.Sensor;
import brooklyn.event.SensorEvent;
import brooklyn.event.SensorEventListener;
import brooklyn.event.basic.BasicConfigKey;
import brooklyn.event.basic.BasicNotificationSensor;
import brooklyn.management.Task;
import brooklyn.management.TaskAdaptable;
import brooklyn.policy.PolicySpec;
import brooklyn.policy.autoscaling.MaxPoolSizeReachedEvent;
import brooklyn.policy.autoscaling.ResizeOperator;
import brooklyn.policy.autoscaling.SizeHistory;
import brooklyn.policy.basic.AbstractPolicy;
import brooklyn.util.JavaGroovyEquivalents;
import brooklyn.util.collections.MutableMap;
import brooklyn.util.flags.SetFromFlag;
import brooklyn.util.flags.TypeCoercions;
import brooklyn.util.task.Tasks;
import brooklyn.util.time.Duration;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.reflect.TypeToken;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import groovy.lang.Closure;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Catalog(name="Auto-scaler", description="Policy that is attached to a Resizable entity and dynamically adjusts its size in response to either keep a metric within a given range, or in response to POOL_COLD and POOL_HOT events")
public class AutoScalerPolicy
extends AbstractPolicy {
    private static final Logger LOG = LoggerFactory.getLogger(AutoScalerPolicy.class);
    public static BasicNotificationSensor<Map> DEFAULT_POOL_HOT_SENSOR;
    public static BasicNotificationSensor<Map> DEFAULT_POOL_COLD_SENSOR;
    public static BasicNotificationSensor<Map> DEFAULT_POOL_OK_SENSOR;
    public static BasicNotificationSensor<MaxPoolSizeReachedEvent> DEFAULT_MAX_SIZE_REACHED_SENSOR;
    public static final String POOL_CURRENT_SIZE_KEY = "pool.current.size";
    public static final String POOL_HIGH_THRESHOLD_KEY = "pool.high.threshold";
    public static final String POOL_LOW_THRESHOLD_KEY = "pool.low.threshold";
    public static final String POOL_CURRENT_WORKRATE_KEY = "pool.current.workrate";
    @SetFromFlag(value="metric")
    public static final ConfigKey<AttributeSensor<? extends Number>> METRIC;
    @SetFromFlag(value="entityWithMetric")
    public static final ConfigKey<Entity> ENTITY_WITH_METRIC;
    @SetFromFlag(value="metricLowerBound")
    public static final ConfigKey<Number> METRIC_LOWER_BOUND;
    @SetFromFlag(value="metricUpperBound")
    public static final ConfigKey<Number> METRIC_UPPER_BOUND;
    @SetFromFlag(value="resizeUpIterationIncrement")
    public static final ConfigKey<Integer> RESIZE_UP_ITERATION_INCREMENT;
    @SetFromFlag(value="resizeUpIterationMax")
    public static final ConfigKey<Integer> RESIZE_UP_ITERATION_MAX;
    @SetFromFlag(value="resizeDownIterationIncrement")
    public static final ConfigKey<Integer> RESIZE_DOWN_ITERATION_INCREMENT;
    @SetFromFlag(value="resizeDownIterationMax")
    public static final ConfigKey<Integer> RESIZE_DOWN_ITERATION_MAX;
    @SetFromFlag(value="minPeriodBetweenExecs")
    public static final ConfigKey<Duration> MIN_PERIOD_BETWEEN_EXECS;
    @SetFromFlag(value="resizeUpStabilizationDelay")
    public static final ConfigKey<Duration> RESIZE_UP_STABILIZATION_DELAY;
    @SetFromFlag(value="resizeDownStabilizationDelay")
    public static final ConfigKey<Duration> RESIZE_DOWN_STABILIZATION_DELAY;
    @SetFromFlag(value="minPoolSize")
    public static final ConfigKey<Integer> MIN_POOL_SIZE;
    @SetFromFlag(value="maxPoolSize")
    public static final ConfigKey<Integer> MAX_POOL_SIZE;
    @SetFromFlag(value="resizeOperator")
    public static final ConfigKey<ResizeOperator> RESIZE_OPERATOR;
    @SetFromFlag(value="currentSizeOperator")
    public static final ConfigKey<Function<Entity, Integer>> CURRENT_SIZE_OPERATOR;
    @SetFromFlag(value="poolHotSensor")
    public static final ConfigKey<BasicNotificationSensor<? extends Map>> POOL_HOT_SENSOR;
    @SetFromFlag(value="poolColdSensor")
    public static final ConfigKey<BasicNotificationSensor<? extends Map>> POOL_COLD_SENSOR;
    @SetFromFlag(value="poolOkSensor")
    public static final ConfigKey<BasicNotificationSensor<? extends Map>> POOL_OK_SENSOR;
    @SetFromFlag(value="maxSizeReachedSensor")
    public static final ConfigKey<BasicNotificationSensor<? super MaxPoolSizeReachedEvent>> MAX_SIZE_REACHED_SENSOR;
    @SetFromFlag(value="maxReachedNotificationDelay")
    public static final ConfigKey<Duration> MAX_REACHED_NOTIFICATION_DELAY;
    private Entity poolEntity;
    private final AtomicBoolean executorQueued = new AtomicBoolean(false);
    private volatile long executorTime = 0L;
    private volatile ScheduledExecutorService executor;
    private SizeHistory recentUnboundedResizes;
    private SizeHistory recentDesiredResizes;
    private long maxReachedLastNotifiedTime;
    private final SensorEventListener<Map> utilizationEventHandler = new SensorEventListener<Map>(){

        public void onEvent(SensorEvent<Map> event) {
            Map properties = (Map)event.getValue();
            Sensor sensor = event.getSensor();
            if (sensor.equals(AutoScalerPolicy.this.getPoolColdSensor())) {
                AutoScalerPolicy.this.onPoolCold(properties);
            } else if (sensor.equals(AutoScalerPolicy.this.getPoolHotSensor())) {
                AutoScalerPolicy.this.onPoolHot(properties);
            } else if (sensor.equals(AutoScalerPolicy.this.getPoolOkSensor())) {
                AutoScalerPolicy.this.onPoolOk(properties);
            } else {
                throw new IllegalStateException("Unexpected sensor type: " + sensor + "; event=" + event);
            }
        }
    };
    private final SensorEventListener<Number> metricEventHandler = new SensorEventListener<Number>(){

        public void onEvent(SensorEvent<Number> event) {
            assert (event.getSensor().equals(AutoScalerPolicy.this.getMetric()));
            AutoScalerPolicy.this.onMetricChanged((Number)event.getValue());
        }
    };

    public static Builder builder() {
        return new Builder();
    }

    public AutoScalerPolicy() {
        this((Map<String, ?>)MutableMap.of());
    }

    public AutoScalerPolicy(Map<String, ?> props) {
        super(props);
    }

    public void init() {
        this.doInit();
    }

    public void rebind() {
        this.doInit();
    }

    protected void doInit() {
        long maxReachedNotificationDelay = this.getMaxReachedNotificationDelay().toMilliseconds();
        this.recentUnboundedResizes = new SizeHistory(maxReachedNotificationDelay);
        long maxResizeStabilizationDelay = Math.max(this.getResizeUpStabilizationDelay().toMilliseconds(), this.getResizeDownStabilizationDelay().toMilliseconds());
        this.recentDesiredResizes = new SizeHistory(maxResizeStabilizationDelay);
        this.executor = Executors.newSingleThreadScheduledExecutor(this.newThreadFactory());
    }

    public void setMetricLowerBound(Number val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing metricLowerBound from {} to {}", new Object[]{this, this.getMetricLowerBound(), val});
        }
        this.config().set(METRIC_LOWER_BOUND, Preconditions.checkNotNull((Object)val));
    }

    public void setMetricUpperBound(Number val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing metricUpperBound from {} to {}", new Object[]{this, this.getMetricUpperBound(), val});
        }
        this.config().set(METRIC_UPPER_BOUND, Preconditions.checkNotNull((Object)val));
    }

    private <T> void setOrDefault(ConfigKey<T> key, T val) {
        if (val == null) {
            val = key.getDefaultValue();
        }
        this.config().set(key, val);
    }

    public int getResizeUpIterationIncrement() {
        return (Integer)this.getConfig(RESIZE_UP_ITERATION_INCREMENT);
    }

    public void setResizeUpIterationIncrement(Integer val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeUpIterationIncrement from {} to {}", new Object[]{this, this.getResizeUpIterationIncrement(), val});
        }
        this.setOrDefault(RESIZE_UP_ITERATION_INCREMENT, val);
    }

    public int getResizeDownIterationIncrement() {
        return (Integer)this.getConfig(RESIZE_DOWN_ITERATION_INCREMENT);
    }

    public void setResizeDownIterationIncrement(Integer val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeDownIterationIncrement from {} to {}", new Object[]{this, this.getResizeDownIterationIncrement(), val});
        }
        this.setOrDefault(RESIZE_DOWN_ITERATION_INCREMENT, val);
    }

    public int getResizeUpIterationMax() {
        return (Integer)this.getConfig(RESIZE_UP_ITERATION_MAX);
    }

    public void setResizeUpIterationMax(Integer val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeUpIterationMax from {} to {}", new Object[]{this, this.getResizeUpIterationMax(), val});
        }
        this.setOrDefault(RESIZE_UP_ITERATION_MAX, val);
    }

    public int getResizeDownIterationMax() {
        return (Integer)this.getConfig(RESIZE_DOWN_ITERATION_MAX);
    }

    public void setResizeDownIterationMax(Integer val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeDownIterationMax from {} to {}", new Object[]{this, this.getResizeDownIterationMax(), val});
        }
        this.setOrDefault(RESIZE_DOWN_ITERATION_MAX, val);
    }

    public void setMinPeriodBetweenExecs(long val) {
        this.setMinPeriodBetweenExecs(Duration.millis((Number)val));
    }

    public void setMinPeriodBetweenExecs(Duration val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing minPeriodBetweenExecs from {} to {}", new Object[]{this, this.getMinPeriodBetweenExecs(), val});
        }
        this.config().set(MIN_PERIOD_BETWEEN_EXECS, (Object)val);
    }

    public void setResizeUpStabilizationDelay(long val) {
        this.setResizeUpStabilizationDelay(Duration.millis((Number)val));
    }

    public void setResizeUpStabilizationDelay(Duration val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeUpStabilizationDelay from {} to {}", new Object[]{this, this.getResizeUpStabilizationDelay(), val});
        }
        this.config().set(RESIZE_UP_STABILIZATION_DELAY, (Object)val);
    }

    public void setResizeDownStabilizationDelay(long val) {
        this.setResizeDownStabilizationDelay(Duration.millis((Number)val));
    }

    public void setResizeDownStabilizationDelay(Duration val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing resizeDownStabilizationDelay from {} to {}", new Object[]{this, this.getResizeDownStabilizationDelay(), val});
        }
        this.config().set(RESIZE_DOWN_STABILIZATION_DELAY, (Object)val);
    }

    public void setMinPoolSize(int val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing minPoolSize from {} to {}", new Object[]{this, this.getMinPoolSize(), val});
        }
        this.config().set(MIN_POOL_SIZE, (Object)val);
    }

    public void setMaxPoolSize(int val) {
        if (LOG.isInfoEnabled()) {
            LOG.info("{} changing maxPoolSize from {} to {}", new Object[]{this, this.getMaxPoolSize(), val});
        }
        this.config().set(MAX_POOL_SIZE, (Object)val);
    }

    private AttributeSensor<? extends Number> getMetric() {
        return (AttributeSensor)this.getConfig(METRIC);
    }

    private Entity getEntityWithMetric() {
        return (Entity)this.getConfig(ENTITY_WITH_METRIC);
    }

    private Number getMetricLowerBound() {
        return (Number)this.getConfig(METRIC_LOWER_BOUND);
    }

    private Number getMetricUpperBound() {
        return (Number)this.getConfig(METRIC_UPPER_BOUND);
    }

    private Duration getMinPeriodBetweenExecs() {
        return (Duration)this.getConfig(MIN_PERIOD_BETWEEN_EXECS);
    }

    private Duration getResizeUpStabilizationDelay() {
        return (Duration)this.getConfig(RESIZE_UP_STABILIZATION_DELAY);
    }

    private Duration getResizeDownStabilizationDelay() {
        return (Duration)this.getConfig(RESIZE_DOWN_STABILIZATION_DELAY);
    }

    private int getMinPoolSize() {
        return (Integer)this.getConfig(MIN_POOL_SIZE);
    }

    private int getMaxPoolSize() {
        return (Integer)this.getConfig(MAX_POOL_SIZE);
    }

    private ResizeOperator getResizeOperator() {
        return (ResizeOperator)this.getConfig(RESIZE_OPERATOR);
    }

    private Function<Entity, Integer> getCurrentSizeOperator() {
        return (Function)this.getConfig(CURRENT_SIZE_OPERATOR);
    }

    private BasicNotificationSensor<? extends Map> getPoolHotSensor() {
        return (BasicNotificationSensor)this.getConfig(POOL_HOT_SENSOR);
    }

    private BasicNotificationSensor<? extends Map> getPoolColdSensor() {
        return (BasicNotificationSensor)this.getConfig(POOL_COLD_SENSOR);
    }

    private BasicNotificationSensor<? extends Map> getPoolOkSensor() {
        return (BasicNotificationSensor)this.getConfig(POOL_OK_SENSOR);
    }

    private BasicNotificationSensor<? super MaxPoolSizeReachedEvent> getMaxSizeReachedSensor() {
        return (BasicNotificationSensor)this.getConfig(MAX_SIZE_REACHED_SENSOR);
    }

    private Duration getMaxReachedNotificationDelay() {
        return (Duration)this.getConfig(MAX_REACHED_NOTIFICATION_DELAY);
    }

    protected <T> void doReconfigureConfig(ConfigKey<T> key, T val) {
        if (key.equals(RESIZE_UP_STABILIZATION_DELAY)) {
            Duration maxResizeStabilizationDelay = Duration.max((Duration)((Duration)val), (Duration)this.getResizeDownStabilizationDelay());
            this.recentDesiredResizes.setWindowSize(maxResizeStabilizationDelay);
        } else if (key.equals(RESIZE_DOWN_STABILIZATION_DELAY)) {
            Duration maxResizeStabilizationDelay = Duration.max((Duration)((Duration)val), (Duration)this.getResizeUpStabilizationDelay());
            this.recentDesiredResizes.setWindowSize(maxResizeStabilizationDelay);
        } else if (!(key.equals(METRIC_LOWER_BOUND) || key.equals(METRIC_UPPER_BOUND) || key.equals(RESIZE_UP_ITERATION_INCREMENT) || key.equals(RESIZE_UP_ITERATION_MAX) || key.equals(RESIZE_DOWN_ITERATION_INCREMENT) || key.equals(RESIZE_DOWN_ITERATION_MAX))) {
            if (key.equals(MIN_POOL_SIZE)) {
                int newMin = (Integer)val;
                if (newMin > (Integer)this.getConfig(MAX_POOL_SIZE)) {
                    throw new IllegalArgumentException("Min pool size " + val + " must not be greater than max pool size " + this.getConfig(MAX_POOL_SIZE));
                }
                this.onPoolSizeLimitsChanged(newMin, (Integer)this.getConfig(MAX_POOL_SIZE));
            } else if (key.equals(MAX_POOL_SIZE)) {
                int newMax = (Integer)val;
                if (newMax < (Integer)this.getConfig(MIN_POOL_SIZE)) {
                    throw new IllegalArgumentException("Min pool size " + val + " must not be greater than max pool size " + this.getConfig(MAX_POOL_SIZE));
                }
                this.onPoolSizeLimitsChanged((Integer)this.getConfig(MIN_POOL_SIZE), newMax);
            } else {
                throw new UnsupportedOperationException("reconfiguring " + key + " unsupported for " + (Object)((Object)this));
            }
        }
    }

    public void suspend() {
        super.suspend();
        if (this.executor != null) {
            this.executor.shutdownNow();
        }
    }

    public void resume() {
        super.resume();
        this.executor = Executors.newSingleThreadScheduledExecutor(this.newThreadFactory());
    }

    public void setEntity(EntityLocal entity) {
        if (!this.config().getRaw(RESIZE_OPERATOR).isPresentAndNonNull()) {
            Preconditions.checkArgument((boolean)(entity instanceof Resizable), (Object)("Provided entity " + entity + " must be an instance of Resizable, because no custom-resizer operator supplied"));
        }
        super.setEntity(entity);
        this.poolEntity = entity;
        if (this.getMetric() != null) {
            EntityLocal entityToSubscribeTo = this.getEntityWithMetric() != null ? this.getEntityWithMetric() : entity;
            this.subscribe((Entity)entityToSubscribeTo, (Sensor)this.getMetric(), (SensorEventListener)this.metricEventHandler);
        }
        this.subscribe(this.poolEntity, (Sensor)this.getPoolColdSensor(), (SensorEventListener)this.utilizationEventHandler);
        this.subscribe(this.poolEntity, (Sensor)this.getPoolHotSensor(), (SensorEventListener)this.utilizationEventHandler);
        this.subscribe(this.poolEntity, (Sensor)this.getPoolOkSensor(), (SensorEventListener)this.utilizationEventHandler);
    }

    private ThreadFactory newThreadFactory() {
        return new ThreadFactoryBuilder().setNameFormat("brooklyn-autoscalerpolicy-%d").build();
    }

    private void onPoolSizeLimitsChanged(final int min, final int max) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} checking pool size on limits changed for {} (between {} and {})", new Object[]{this, this.poolEntity, min, max});
        }
        if (this.isRunning() && this.isEntityUp()) {
            this.executor.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        int currentSize = (Integer)AutoScalerPolicy.this.getCurrentSizeOperator().apply((Object)AutoScalerPolicy.this.entity);
                        int desiredSize = Math.min(max, Math.max(min, currentSize));
                        if (currentSize != desiredSize) {
                            if (LOG.isInfoEnabled()) {
                                LOG.info("{} resizing pool {} immediateley from {} to {} (due to new pool size limits)", new Object[]{this, AutoScalerPolicy.this.poolEntity, currentSize, desiredSize});
                            }
                            AutoScalerPolicy.this.getResizeOperator().resize(AutoScalerPolicy.this.poolEntity, desiredSize);
                        }
                    }
                    catch (Exception e) {
                        if (AutoScalerPolicy.this.isRunning()) {
                            LOG.error("Error resizing: " + e, (Throwable)e);
                        } else if (LOG.isDebugEnabled()) {
                            LOG.debug("Error resizing, but no longer running: " + e, (Throwable)e);
                        }
                    }
                    catch (Throwable t) {
                        LOG.error("Error resizing: " + t, t);
                        throw Throwables.propagate((Throwable)t);
                    }
                }
            });
        }
    }

    private void onMetricChanged(Number val) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} recording pool-metric for {}: {}", new Object[]{this, this.poolEntity, val});
        }
        if (val == null) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} not resizing pool {}, inbound metric is null", new Object[]{this, this.poolEntity});
            }
            return;
        }
        ScalingData data = new ScalingData();
        data.currentMetricValue = val.doubleValue();
        data.currentSize = (Integer)this.getCurrentSizeOperator().apply((Object)this.entity);
        data.metricUpperBound = this.getMetricUpperBound().doubleValue();
        data.metricLowerBound = this.getMetricLowerBound().doubleValue();
        this.analyze(data, "pool");
    }

    private void onPoolCold(Map<String, ?> properties) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} recording pool-cold for {}: {}", new Object[]{this, this.poolEntity, properties});
        }
        this.analyzeOnHotOrColdSensor(ScalingType.COLD, "cold pool", properties);
    }

    private void onPoolHot(Map<String, ?> properties) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} recording pool-hot for {}: {}", new Object[]{this, this.poolEntity, properties});
        }
        this.analyzeOnHotOrColdSensor(ScalingType.HOT, "hot pool", properties);
    }

    private void analyzeOnHotOrColdSensor(ScalingType scalingMode, String description, Map<String, ?> properties) {
        ScalingData data = new ScalingData();
        data.scalingMode = scalingMode;
        data.currentMetricValue = (Double)properties.get(POOL_CURRENT_WORKRATE_KEY);
        data.currentSize = (Integer)properties.get(POOL_CURRENT_SIZE_KEY);
        data.metricUpperBound = (Double)properties.get(POOL_HIGH_THRESHOLD_KEY);
        data.metricLowerBound = (Double)properties.get(POOL_LOW_THRESHOLD_KEY);
        this.analyze(data, description);
    }

    private void analyze(ScalingData data, String description) {
        int desiredSizeUnconstrained;
        if (data.isHot()) {
            desiredSizeUnconstrained = (int)Math.ceil(data.getCurrentTotalActivity() / data.metricUpperBound);
            data.scalingMode = ScalingType.HOT;
        } else if (data.isCold()) {
            desiredSizeUnconstrained = (int)Math.floor(data.getCurrentTotalActivity() / data.metricLowerBound);
            data.scalingMode = ScalingType.COLD;
        } else {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} not resizing pool {} from {} ({} within range {}..{})", new Object[]{this, this.poolEntity, data.currentSize, data.currentMetricValue, data.metricLowerBound, data.metricUpperBound});
            }
            this.abortResize(data.currentSize);
            return;
        }
        if (LOG.isTraceEnabled()) {
            LOG.debug("{} detected unconstrained desired size {}", new Object[]{this, desiredSizeUnconstrained});
        }
        int desiredSize = this.applyMinMaxConstraints(desiredSizeUnconstrained);
        if (data.scalingMode == ScalingType.COLD && desiredSize < data.currentSize) {
            int delta = data.currentSize - desiredSize;
            int scaleIncrement = this.getResizeDownIterationIncrement();
            int scaleMax = this.getResizeDownIterationMax();
            if (delta > scaleMax) {
                delta = scaleMax;
            } else if (delta % scaleIncrement != 0) {
                delta += scaleIncrement - delta % scaleIncrement;
            }
            if (data.metricUpperBound != null) {
                for (desiredSize = data.currentSize - delta; desiredSize < data.currentSize && data.getCurrentTotalActivity() > data.metricUpperBound * (double)desiredSize; desiredSize += scaleIncrement) {
                    if (!LOG.isTraceEnabled()) continue;
                    LOG.trace("{} when resizing back pool {} from {}, tweaking from {} to prevent thrashing", new Object[]{this, this.poolEntity, data.currentSize, desiredSize});
                }
            }
            if ((desiredSize = this.applyMinMaxConstraints(desiredSize)) >= data.currentSize) {
                data.scalingMode = null;
            }
        } else if (data.scalingMode == ScalingType.HOT && desiredSize > data.currentSize) {
            int delta = desiredSize - data.currentSize;
            int scaleIncrement = this.getResizeUpIterationIncrement();
            int scaleMax = this.getResizeUpIterationMax();
            if (delta > scaleMax) {
                delta = scaleMax;
            } else if (delta % scaleIncrement != 0) {
                delta += scaleIncrement - delta % scaleIncrement;
            }
            desiredSize = data.currentSize + delta;
            desiredSize = this.applyMinMaxConstraints(desiredSize);
            if (desiredSize <= data.currentSize) {
                data.scalingMode = null;
            }
        } else {
            data.scalingMode = null;
        }
        if (data.scalingMode != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} provisionally resizing {} {} from {} to {} ({} < {}; ideal size {})", new Object[]{this, description, this.poolEntity, data.currentSize, desiredSize, data.currentMetricValue, data.metricLowerBound, desiredSizeUnconstrained});
            }
            this.scheduleResize(desiredSize);
        } else {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} not resizing {} {} from {} to {}, {} out of healthy range {}..{} but unconstrained size {} blocked by bounds/check", new Object[]{this, description, this.poolEntity, data.currentSize, desiredSize, data.currentMetricValue, data.metricLowerBound, data.metricUpperBound, desiredSizeUnconstrained});
            }
            this.abortResize(data.currentSize);
        }
        this.onNewUnboundedPoolSize(desiredSizeUnconstrained);
    }

    private int applyMinMaxConstraints(int desiredSize) {
        desiredSize = Math.max(this.getMinPoolSize(), desiredSize);
        desiredSize = Math.min(this.getMaxPoolSize(), desiredSize);
        return desiredSize;
    }

    private void onPoolOk(Map<String, ?> properties) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} recording pool-ok for {}: {}", new Object[]{this, this.poolEntity, properties});
        }
        int poolCurrentSize = (Integer)properties.get(POOL_CURRENT_SIZE_KEY);
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} not resizing ok pool {} from {}", new Object[]{this, this.poolEntity, poolCurrentSize});
        }
        this.abortResize(poolCurrentSize);
    }

    private void scheduleResize(int newSize) {
        this.recentDesiredResizes.add(newSize);
        this.scheduleResize();
    }

    private void onNewUnboundedPoolSize(int val) {
        if (this.getMaxSizeReachedSensor() != null) {
            this.recentUnboundedResizes.add(val);
            this.scheduleResize();
        }
    }

    private void abortResize(int currentSize) {
        this.recentDesiredResizes.add(currentSize);
        this.recentUnboundedResizes.add(currentSize);
    }

    private boolean isEntityUp() {
        if (this.entity == null) {
            return false;
        }
        if (this.entity.getEntityType().getSensors().contains(Startable.SERVICE_UP)) {
            return Boolean.TRUE.equals(this.entity.getAttribute(Startable.SERVICE_UP));
        }
        return true;
    }

    private void scheduleResize() {
        if (this.isRunning() && this.isEntityUp() && this.executorQueued.compareAndSet(false, true)) {
            long now = System.currentTimeMillis();
            long delay = Math.max(0L, this.executorTime + this.getMinPeriodBetweenExecs().toMilliseconds() - now);
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} scheduling resize in {}ms", (Object)this, (Object)delay);
            }
            this.executor.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        AutoScalerPolicy.this.executorTime = System.currentTimeMillis();
                        AutoScalerPolicy.this.executorQueued.set(false);
                        AutoScalerPolicy.this.resizeNow();
                        AutoScalerPolicy.this.notifyMaxReachedIfRequiredNow();
                    }
                    catch (Exception e) {
                        if (AutoScalerPolicy.this.isRunning()) {
                            LOG.error("Error resizing: " + e, (Throwable)e);
                        } else if (LOG.isDebugEnabled()) {
                            LOG.debug("Error resizing, but no longer running: " + e, (Throwable)e);
                        }
                    }
                    catch (Throwable t) {
                        LOG.error("Error resizing: " + t, t);
                        throw Throwables.propagate((Throwable)t);
                    }
                }
            }, delay, TimeUnit.MILLISECONDS);
        }
    }

    private void notifyMaxReachedIfRequiredNow() {
        BasicNotificationSensor<? super MaxPoolSizeReachedEvent> maxSizeReachedSensor = this.getMaxSizeReachedSensor();
        if (maxSizeReachedSensor == null) {
            return;
        }
        SizeHistory.WindowSummary valsSummary = this.recentUnboundedResizes.summarizeWindow(this.getMaxReachedNotificationDelay());
        long timeWindowSize = this.getMaxReachedNotificationDelay().toMilliseconds();
        long currentPoolSize = ((Integer)this.getCurrentSizeOperator().apply((Object)this.poolEntity)).intValue();
        int maxAllowedPoolSize = this.getMaxPoolSize();
        long unboundedSustainedMaxPoolSize = valsSummary.min;
        long unboundedCurrentPoolSize = valsSummary.latest;
        if (this.maxReachedLastNotifiedTime <= 0L) {
            if (unboundedSustainedMaxPoolSize > (long)maxAllowedPoolSize) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} notifying listener of max pool size reached; current {}, max {}, unbounded current {}, unbounded max {}", new Object[]{this, currentPoolSize, maxAllowedPoolSize, unboundedCurrentPoolSize, unboundedSustainedMaxPoolSize});
                }
                this.maxReachedLastNotifiedTime = System.currentTimeMillis();
                MaxPoolSizeReachedEvent event = MaxPoolSizeReachedEvent.builder().currentPoolSize(currentPoolSize).maxAllowed(maxAllowedPoolSize).currentUnbounded(unboundedCurrentPoolSize).maxUnbounded(unboundedSustainedMaxPoolSize).timeWindow(timeWindowSize).build();
                this.entity.emit(maxSizeReachedSensor, (Object)event);
            } else if (valsSummary.max > (long)maxAllowedPoolSize) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("{} re-scheduling max-reached check for {}, as unbounded size not stable (min {}, max {}, latest {})", new Object[]{this, this.poolEntity, valsSummary.min, valsSummary.max, valsSummary.latest});
                }
                this.scheduleResize();
            }
        }
    }

    private void resizeNow() {
        long currentPoolSize = ((Integer)this.getCurrentSizeOperator().apply((Object)this.poolEntity)).intValue();
        CalculatedDesiredPoolSize calculatedDesiredPoolSize = this.calculateDesiredPoolSize(currentPoolSize);
        final long desiredPoolSize = calculatedDesiredPoolSize.size;
        boolean stable = calculatedDesiredPoolSize.stable;
        if (!stable) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} re-scheduling resize check for {}, as desired size not stable (current {}, desired {}); continuing with resize...", new Object[]{this, this.poolEntity, currentPoolSize, desiredPoolSize});
            }
            this.scheduleResize();
        }
        if (currentPoolSize == desiredPoolSize) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} not resizing pool {} from {} to {}", new Object[]{this, this.poolEntity, currentPoolSize, desiredPoolSize});
            }
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} requesting resize to {}; current {}, min {}, max {}", new Object[]{this, desiredPoolSize, currentPoolSize, this.getMinPoolSize(), this.getMaxPoolSize()});
        }
        ((Task)Entities.submit((Entity)this.entity, (TaskAdaptable)Tasks.builder().name("Auto-scaler").description("Auto-scaler recommending resize from " + currentPoolSize + " to " + desiredPoolSize).tag((Object)"NON-TRANSIENT").body((Callable)new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                AutoScalerPolicy.this.getResizeOperator().resize(AutoScalerPolicy.this.poolEntity, (int)desiredPoolSize);
                return null;
            }
        }).build())).blockUntilEnded();
    }

    private CalculatedDesiredPoolSize calculateDesiredPoolSize(long currentPoolSize) {
        boolean stable;
        long desiredPoolSize;
        long now = System.currentTimeMillis();
        SizeHistory.WindowSummary downsizeSummary = this.recentDesiredResizes.summarizeWindow(this.getResizeDownStabilizationDelay());
        SizeHistory.WindowSummary upsizeSummary = this.recentDesiredResizes.summarizeWindow(this.getResizeUpStabilizationDelay());
        long maxDesiredPoolSize = upsizeSummary.min;
        boolean stableForGrowing = upsizeSummary.stableForGrowth;
        long minDesiredPoolSize = downsizeSummary.max;
        boolean stableForShrinking = downsizeSummary.stableForShrinking;
        if (currentPoolSize < maxDesiredPoolSize) {
            desiredPoolSize = maxDesiredPoolSize;
            stable = stableForGrowing;
        } else if (currentPoolSize > minDesiredPoolSize) {
            desiredPoolSize = minDesiredPoolSize;
            stable = stableForShrinking;
        } else {
            desiredPoolSize = currentPoolSize;
            boolean bl = stable = stableForGrowing && stableForShrinking;
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} calculated desired pool size: from {} to {}; minDesired {}, maxDesired {}; stable {}; now {}; downsizeHistory {}; upsizeHistory {}", new Object[]{this, currentPoolSize, desiredPoolSize, minDesiredPoolSize, maxDesiredPoolSize, stable, now, downsizeSummary, upsizeSummary});
        }
        return new CalculatedDesiredPoolSize(desiredPoolSize, stable);
    }

    public String toString() {
        return ((Object)((Object)this)).getClass().getSimpleName() + (JavaGroovyEquivalents.groovyTruth((String)this.name) ? "(" + this.name + ")" : "");
    }

    static {
        TypeCoercions.registerAdapter(Closure.class, ResizeOperator.class, (Function)new Function<Closure, ResizeOperator>(){

            public ResizeOperator apply(final Closure closure) {
                return new ResizeOperator(){

                    @Override
                    public Integer resize(Entity entity, Integer input) {
                        return (Integer)closure.call(new Object[]{entity, input});
                    }
                };
            }
        });
        DEFAULT_POOL_HOT_SENSOR = new BasicNotificationSensor(Map.class, "resizablepool.hot", "Pool is over-utilized; it has insufficient resource for current workload");
        DEFAULT_POOL_COLD_SENSOR = new BasicNotificationSensor(Map.class, "resizablepool.cold", "Pool is under-utilized; it has too much resource for current workload");
        DEFAULT_POOL_OK_SENSOR = new BasicNotificationSensor(Map.class, "resizablepool.cold", "Pool utilization is ok; the available resources are fine for the current workload");
        DEFAULT_MAX_SIZE_REACHED_SENSOR = new BasicNotificationSensor(MaxPoolSizeReachedEvent.class, "resizablepool.maxSizeReached", "Consistently wanted to resize the pool above the max allowed size");
        METRIC = BasicConfigKey.builder((TypeToken)new TypeToken<AttributeSensor<? extends Number>>(){}).name("autoscaler.metric").build();
        ENTITY_WITH_METRIC = BasicConfigKey.builder(Entity.class).name("autoscaler.entityWithMetric").build();
        METRIC_LOWER_BOUND = BasicConfigKey.builder(Number.class).name("autoscaler.metricLowerBound").reconfigurable(true).build();
        METRIC_UPPER_BOUND = BasicConfigKey.builder(Number.class).name("autoscaler.metricUpperBound").reconfigurable(true).build();
        RESIZE_UP_ITERATION_INCREMENT = BasicConfigKey.builder(Integer.class).name("autoscaler.resizeUpIterationIncrement").description("Batch size for resizing up; the size will be increased by a multiple of this value").defaultValue((Object)1).reconfigurable(true).build();
        RESIZE_UP_ITERATION_MAX = BasicConfigKey.builder(Integer.class).name("autoscaler.resizeUpIterationMax").defaultValue((Object)Integer.MAX_VALUE).description("Maximum change to the size on a single iteration when scaling up").reconfigurable(true).build();
        RESIZE_DOWN_ITERATION_INCREMENT = BasicConfigKey.builder(Integer.class).name("autoscaler.resizeDownIterationIncrement").description("Batch size for resizing down; the size will be decreased by a multiple of this value").defaultValue((Object)1).reconfigurable(true).build();
        RESIZE_DOWN_ITERATION_MAX = BasicConfigKey.builder(Integer.class).name("autoscaler.resizeDownIterationMax").defaultValue((Object)Integer.MAX_VALUE).description("Maximum change to the size on a single iteration when scaling down").reconfigurable(true).build();
        MIN_PERIOD_BETWEEN_EXECS = BasicConfigKey.builder(Duration.class).name("autoscaler.minPeriodBetweenExecs").defaultValue((Object)Duration.millis((Number)100)).build();
        RESIZE_UP_STABILIZATION_DELAY = BasicConfigKey.builder(Duration.class).name("autoscaler.resizeUpStabilizationDelay").defaultValue((Object)Duration.ZERO).reconfigurable(true).build();
        RESIZE_DOWN_STABILIZATION_DELAY = BasicConfigKey.builder(Duration.class).name("autoscaler.resizeDownStabilizationDelay").defaultValue((Object)Duration.ZERO).reconfigurable(true).build();
        MIN_POOL_SIZE = BasicConfigKey.builder(Integer.class).name("autoscaler.minPoolSize").defaultValue((Object)1).reconfigurable(true).build();
        MAX_POOL_SIZE = BasicConfigKey.builder(Integer.class).name("autoscaler.maxPoolSize").defaultValue((Object)Integer.MAX_VALUE).reconfigurable(true).build();
        RESIZE_OPERATOR = BasicConfigKey.builder(ResizeOperator.class).name("autoscaler.resizeOperator").defaultValue((Object)new ResizeOperator(){

            @Override
            public Integer resize(Entity entity, Integer desiredSize) {
                return ((Resizable)entity).resize(desiredSize);
            }
        }).build();
        CURRENT_SIZE_OPERATOR = BasicConfigKey.builder((TypeToken)new TypeToken<Function<Entity, Integer>>(){}).name("autoscaler.currentSizeOperator").defaultValue((Object)new Function<Entity, Integer>(){

            public Integer apply(Entity entity) {
                return ((Resizable)entity).getCurrentSize();
            }
        }).build();
        POOL_HOT_SENSOR = BasicConfigKey.builder((TypeToken)new TypeToken<BasicNotificationSensor<? extends Map>>(){}).name("autoscaler.poolHotSensor").defaultValue(DEFAULT_POOL_HOT_SENSOR).build();
        POOL_COLD_SENSOR = BasicConfigKey.builder((TypeToken)new TypeToken<BasicNotificationSensor<? extends Map>>(){}).name("autoscaler.poolColdSensor").defaultValue(DEFAULT_POOL_COLD_SENSOR).build();
        POOL_OK_SENSOR = BasicConfigKey.builder((TypeToken)new TypeToken<BasicNotificationSensor<? extends Map>>(){}).name("autoscaler.poolOkSensor").defaultValue(DEFAULT_POOL_OK_SENSOR).build();
        MAX_SIZE_REACHED_SENSOR = BasicConfigKey.builder((TypeToken)new TypeToken<BasicNotificationSensor<? super MaxPoolSizeReachedEvent>>(){}).name("autoscaler.maxSizeReachedSensor").description("Sensor for which a notification will be emitted (on the associated entity) when we consistently wanted to resize the pool above the max allowed size, for maxReachedNotificationDelay milliseconds").build();
        MAX_REACHED_NOTIFICATION_DELAY = BasicConfigKey.builder(Duration.class).name("autoscaler.maxReachedNotificationDelay").description("Time that we consistently wanted to go above the maxPoolSize for, after which the maxSizeReachedSensor (if any) will be emitted").defaultValue((Object)Duration.ZERO).build();
    }

    private static class CalculatedDesiredPoolSize {
        final long size;
        final boolean stable;

        CalculatedDesiredPoolSize(long size, boolean stable) {
            this.size = size;
            this.stable = stable;
        }
    }

    private static class ScalingData {
        ScalingType scalingMode;
        int currentSize;
        double currentMetricValue;
        Double metricUpperBound;
        Double metricLowerBound;

        private ScalingData() {
        }

        public double getCurrentTotalActivity() {
            return this.currentMetricValue * (double)this.currentSize;
        }

        public boolean isHot() {
            return (this.scalingMode == null || this.scalingMode == ScalingType.HOT) && this.isValid(this.metricUpperBound) && this.currentMetricValue > this.metricUpperBound;
        }

        public boolean isCold() {
            return (this.scalingMode == null || this.scalingMode == ScalingType.COLD) && this.isValid(this.metricLowerBound) && this.currentMetricValue < this.metricLowerBound;
        }

        private boolean isValid(Double bound) {
            return bound != null && bound > 0.0;
        }
    }

    private static enum ScalingType {
        HOT,
        COLD;

    }

    public static class Builder {
        private String id;
        private String name;
        private AttributeSensor<? extends Number> metric;
        private Entity entityWithMetric;
        private Number metricUpperBound;
        private Number metricLowerBound;
        private int minPoolSize = 1;
        private int maxPoolSize = Integer.MAX_VALUE;
        private Integer resizeDownIterationIncrement;
        private Integer resizeDownIterationMax;
        private Integer resizeUpIterationIncrement;
        private Integer resizeUpIterationMax;
        private Duration minPeriodBetweenExecs;
        private Duration resizeUpStabilizationDelay;
        private Duration resizeDownStabilizationDelay;
        private ResizeOperator resizeOperator;
        private Function<Entity, Integer> currentSizeOperator;
        private BasicNotificationSensor<?> poolHotSensor;
        private BasicNotificationSensor<?> poolColdSensor;
        private BasicNotificationSensor<?> poolOkSensor;
        private BasicNotificationSensor<? super MaxPoolSizeReachedEvent> maxSizeReachedSensor;
        private Duration maxReachedNotificationDelay;

        public Builder id(String val) {
            this.id = val;
            return this;
        }

        public Builder name(String val) {
            this.name = val;
            return this;
        }

        public Builder metric(AttributeSensor<? extends Number> val) {
            this.metric = val;
            return this;
        }

        public Builder entityWithMetric(Entity val) {
            this.entityWithMetric = val;
            return this;
        }

        public Builder metricLowerBound(Number val) {
            this.metricLowerBound = val;
            return this;
        }

        public Builder metricUpperBound(Number val) {
            this.metricUpperBound = val;
            return this;
        }

        public Builder metricRange(Number min, Number max) {
            this.metricLowerBound = (Number)Preconditions.checkNotNull((Object)min);
            this.metricUpperBound = (Number)Preconditions.checkNotNull((Object)max);
            return this;
        }

        public Builder minPoolSize(int val) {
            this.minPoolSize = val;
            return this;
        }

        public Builder maxPoolSize(int val) {
            this.maxPoolSize = val;
            return this;
        }

        public Builder sizeRange(int min, int max) {
            this.minPoolSize = min;
            this.maxPoolSize = max;
            return this;
        }

        public Builder resizeUpIterationIncrement(Integer val) {
            this.resizeUpIterationIncrement = val;
            return this;
        }

        public Builder resizeUpIterationMax(Integer val) {
            this.resizeUpIterationMax = val;
            return this;
        }

        public Builder resizeDownIterationIncrement(Integer val) {
            this.resizeUpIterationIncrement = val;
            return this;
        }

        public Builder resizeDownIterationMax(Integer val) {
            this.resizeUpIterationMax = val;
            return this;
        }

        @Deprecated
        public Builder minPeriodBetweenExecs(long val) {
            return this.minPeriodBetweenExecs(Duration.of((long)val, (TimeUnit)TimeUnit.MILLISECONDS));
        }

        public Builder minPeriodBetweenExecs(Duration val) {
            this.minPeriodBetweenExecs = val;
            return this;
        }

        @Deprecated
        public Builder resizeUpStabilizationDelay(long val) {
            return this.resizeUpStabilizationDelay(Duration.of((long)val, (TimeUnit)TimeUnit.MILLISECONDS));
        }

        public Builder resizeUpStabilizationDelay(Duration val) {
            this.resizeUpStabilizationDelay = val;
            return this;
        }

        @Deprecated
        public Builder resizeDownStabilizationDelay(long val) {
            return this.resizeDownStabilizationDelay(Duration.of((long)val, (TimeUnit)TimeUnit.MILLISECONDS));
        }

        public Builder resizeDownStabilizationDelay(Duration val) {
            this.resizeDownStabilizationDelay = val;
            return this;
        }

        public Builder resizeOperator(ResizeOperator val) {
            this.resizeOperator = val;
            return this;
        }

        public Builder currentSizeOperator(Function<Entity, Integer> val) {
            this.currentSizeOperator = val;
            return this;
        }

        public Builder poolHotSensor(BasicNotificationSensor<?> val) {
            this.poolHotSensor = val;
            return this;
        }

        public Builder poolColdSensor(BasicNotificationSensor<?> val) {
            this.poolColdSensor = val;
            return this;
        }

        public Builder poolOkSensor(BasicNotificationSensor<?> val) {
            this.poolOkSensor = val;
            return this;
        }

        public Builder maxSizeReachedSensor(BasicNotificationSensor<? super MaxPoolSizeReachedEvent> val) {
            this.maxSizeReachedSensor = val;
            return this;
        }

        @Deprecated
        public Builder maxReachedNotificationDelay(long val) {
            return this.maxReachedNotificationDelay(Duration.of((long)val, (TimeUnit)TimeUnit.MILLISECONDS));
        }

        public Builder maxReachedNotificationDelay(Duration val) {
            this.maxReachedNotificationDelay = val;
            return this;
        }

        public AutoScalerPolicy build() {
            return new AutoScalerPolicy(this.toFlags());
        }

        public PolicySpec<AutoScalerPolicy> buildSpec() {
            return PolicySpec.create(AutoScalerPolicy.class).configure(this.toFlags());
        }

        private Map<String, ?> toFlags() {
            return MutableMap.builder().putIfNotNull((Object)"id", (Object)this.id).putIfNotNull((Object)"name", (Object)this.name).putIfNotNull((Object)"metric", this.metric).putIfNotNull((Object)"entityWithMetric", (Object)this.entityWithMetric).putIfNotNull((Object)"metricUpperBound", (Object)this.metricUpperBound).putIfNotNull((Object)"metricLowerBound", (Object)this.metricLowerBound).putIfNotNull((Object)"minPoolSize", (Object)this.minPoolSize).putIfNotNull((Object)"maxPoolSize", (Object)this.maxPoolSize).putIfNotNull((Object)"resizeUpIterationMax", (Object)this.resizeUpIterationMax).putIfNotNull((Object)"resizeUpIterationIncrement", (Object)this.resizeUpIterationIncrement).putIfNotNull((Object)"resizeDownIterationMax", (Object)this.resizeDownIterationMax).putIfNotNull((Object)"resizeDownIterationIncrement", (Object)this.resizeDownIterationIncrement).putIfNotNull((Object)"minPeriodBetweenExecs", (Object)this.minPeriodBetweenExecs).putIfNotNull((Object)"resizeUpStabilizationDelay", (Object)this.resizeUpStabilizationDelay).putIfNotNull((Object)"resizeDownStabilizationDelay", (Object)this.resizeDownStabilizationDelay).putIfNotNull((Object)"resizeOperator", (Object)this.resizeOperator).putIfNotNull((Object)"currentSizeOperator", this.currentSizeOperator).putIfNotNull((Object)"poolHotSensor", this.poolHotSensor).putIfNotNull((Object)"poolColdSensor", this.poolColdSensor).putIfNotNull((Object)"poolOkSensor", this.poolOkSensor).putIfNotNull((Object)"maxSizeReachedSensor", this.maxSizeReachedSensor).putIfNotNull((Object)"maxReachedNotificationDelay", (Object)this.maxReachedNotificationDelay).build();
        }
    }
}

