/*
 * Decompiled with CFR 0.152.
 */
package com.thinkaurelius.titan.diskstorage.locking;

import com.google.common.base.Preconditions;
import com.thinkaurelius.titan.diskstorage.util.KeyColumn;
import com.thinkaurelius.titan.diskstorage.util.time.Timepoint;
import com.thinkaurelius.titan.diskstorage.util.time.TimestampProvider;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalLockMediator<T> {
    private static final Logger log = LoggerFactory.getLogger(LocalLockMediator.class);
    private final String name;
    private final TimestampProvider times;
    private DelayQueue<ExpirableKeyColumn> expiryQueue = new DelayQueue();
    private ExecutorService lockCleanerService = Executors.newFixedThreadPool(1, new ThreadFactory(){

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = Executors.defaultThreadFactory().newThread(runnable);
            thread.setDaemon(true);
            return thread;
        }
    });
    private final ConcurrentHashMap<KeyColumn, AuditRecord<T>> locks = new ConcurrentHashMap();

    public LocalLockMediator(String name, TimestampProvider times) {
        this.name = name;
        this.times = times;
        Preconditions.checkNotNull((Object)name);
        Preconditions.checkNotNull((Object)times);
        this.lockCleanerService.submit(new LockCleaner());
    }

    public boolean lock(KeyColumn kc, T requestor, Timepoint expires) {
        assert (null != kc);
        assert (null != requestor);
        AuditRecord audit = new AuditRecord(requestor, expires);
        AuditRecord inmap = this.locks.putIfAbsent(kc, audit);
        boolean success = false;
        if (null == inmap) {
            if (log.isTraceEnabled()) {
                log.trace("New local lock created: {} namespace={} txn={}", new Object[]{kc, this.name, requestor});
            }
            success = true;
        } else if (inmap.equals(audit)) {
            success = this.locks.replace(kc, inmap, audit);
            if (log.isTraceEnabled()) {
                if (success) {
                    log.trace("Updated local lock expiration: {} namespace={} txn={} oldexp={} newexp={}", new Object[]{kc, this.name, requestor, inmap.expires, audit.expires});
                } else {
                    log.trace("Failed to update local lock expiration: {} namespace={} txn={} oldexp={} newexp={}", new Object[]{kc, this.name, requestor, inmap.expires, audit.expires});
                }
            }
        } else if (0 > inmap.expires.compareTo((Object)this.times.getTime())) {
            success = this.locks.replace(kc, inmap, audit);
            if (log.isTraceEnabled()) {
                log.trace("Discarding expired lock: {} namespace={} txn={} expired={}", new Object[]{kc, this.name, inmap.holder, inmap.expires});
            }
        } else if (log.isTraceEnabled()) {
            log.trace("Local lock failed: {} namespace={} txn={} (already owned by {})", new Object[]{kc, this.name, requestor, inmap});
        }
        if (success) {
            this.expiryQueue.add(new ExpirableKeyColumn(kc, expires));
        }
        return success;
    }

    public boolean unlock(KeyColumn kc, T requestor) {
        if (!this.locks.containsKey(kc)) {
            log.info("Local unlock failed: no locks found for {}", (Object)kc);
            return false;
        }
        AuditRecord unlocker = new AuditRecord(requestor, null);
        AuditRecord<T> holder = this.locks.get(kc);
        if (!holder.equals(unlocker)) {
            log.error("Local unlock of {} by {} failed: it is held by {}", new Object[]{kc, unlocker, holder});
            return false;
        }
        boolean removed = this.locks.remove(kc, unlocker);
        if (removed) {
            this.expiryQueue.remove(kc);
            if (log.isTraceEnabled()) {
                log.trace("Local unlock succeeded: {} namespace={} txn={}", new Object[]{kc, this.name, requestor});
            }
        } else {
            log.warn("Local unlock warning: lock record for {} disappeared during removal; this suggests the lock either expired while we were removing it, or that it was erroneously unlocked multiple times.", (Object)kc);
        }
        return true;
    }

    public String toString() {
        return "LocalLockMediator [" + this.name + ",  ~" + this.locks.size() + " current locks]";
    }

    private static class ExpirableKeyColumn
    implements Delayed {
        private Timepoint expiryTime;
        private KeyColumn kc;

        public ExpirableKeyColumn(KeyColumn keyColumn, Timepoint expiryTime) {
            this.kc = keyColumn;
            this.expiryTime = expiryTime;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return this.expiryTime.getTimestamp(TimeUnit.NANOSECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            if (this.expiryTime.getTimestamp(TimeUnit.NANOSECONDS) < ((ExpirableKeyColumn)o).expiryTime.getTimestamp(TimeUnit.NANOSECONDS)) {
                return -1;
            }
            if (this.expiryTime.getTimestamp(TimeUnit.NANOSECONDS) > ((ExpirableKeyColumn)o).expiryTime.getTimestamp(TimeUnit.NANOSECONDS)) {
                return 1;
            }
            return 0;
        }

        public KeyColumn getKeyColumn() {
            return this.kc;
        }
    }

    private class LockCleaner
    implements Runnable {
        private LockCleaner() {
        }

        @Override
        public void run() {
            try {
                while (true) {
                    log.debug("Lock Cleaner service started");
                    ExpirableKeyColumn lock = (ExpirableKeyColumn)LocalLockMediator.this.expiryQueue.take();
                    log.debug("Expiring key column " + lock.getKeyColumn());
                    LocalLockMediator.this.locks.remove(lock.getKeyColumn());
                }
            }
            catch (InterruptedException e) {
                log.debug("Received interrupt. Exiting");
                return;
            }
        }
    }

    private static class AuditRecord<T> {
        private final T holder;
        private final Timepoint expires;
        private int hashCode;

        private AuditRecord(T holder, Timepoint expires) {
            this.holder = holder;
            this.expires = expires;
        }

        public int hashCode() {
            if (0 == this.hashCode) {
                this.hashCode = this.holder.hashCode();
            }
            return this.hashCode;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            AuditRecord other = (AuditRecord)obj;
            return !(this.holder == null ? other.holder != null : !this.holder.equals(other.holder));
        }

        public String toString() {
            return "AuditRecord [txn=" + this.holder + ", expires=" + this.expires + "]";
        }
    }
}

