/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
*/
package org.apache.airavata.gfac.gram.handler;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.*;

import org.apache.airavata.common.exception.ApplicationSettingsException;
import org.apache.airavata.common.utils.StringUtil;
import org.apache.airavata.commons.gfac.type.ActualParameter;
import org.apache.airavata.commons.gfac.type.ApplicationDescription;
import org.apache.airavata.commons.gfac.type.MappingFactory;
import org.apache.airavata.gfac.GFacException;
import org.apache.airavata.gfac.ToolsException;
import org.apache.airavata.gfac.core.context.JobExecutionContext;
import org.apache.airavata.gfac.core.context.MessageContext;
import org.apache.airavata.gfac.core.handler.AbstractHandler;
import org.apache.airavata.gfac.core.handler.GFacHandlerException;
import org.apache.airavata.gfac.core.provider.GFacProviderException;
import org.apache.airavata.gfac.core.utils.GFacUtils;
import org.apache.airavata.gfac.core.utils.OutputUtils;
import org.apache.airavata.gfac.gram.security.GSISecurityContext;
import org.apache.airavata.gfac.gram.external.GridFtp;
import org.apache.airavata.gfac.gram.util.GramProviderUtils;
import org.apache.airavata.model.workspace.experiment.CorrectiveAction;
import org.apache.airavata.model.workspace.experiment.DataObjectType;
import org.apache.airavata.model.workspace.experiment.DataTransferDetails;
import org.apache.airavata.model.workspace.experiment.ErrorCategory;
import org.apache.airavata.model.workspace.experiment.TaskDetails;
import org.apache.airavata.model.workspace.experiment.TransferState;
import org.apache.airavata.model.workspace.experiment.TransferStatus;
import org.apache.airavata.registry.cpi.ChildDataType;
import org.apache.airavata.registry.cpi.Registry;
import org.apache.airavata.schemas.gfac.ApplicationDeploymentDescriptionType;
import org.apache.airavata.schemas.gfac.GlobusHostType;
import org.apache.airavata.schemas.gfac.HostDescriptionType;
import org.apache.airavata.schemas.gfac.StringArrayType;
import org.apache.airavata.schemas.gfac.URIArrayType;
import org.apache.airavata.schemas.gfac.URIParameterType;
import org.apache.airavata.schemas.gfac.UnicoreHostType;
import org.ietf.jgss.GSSCredential;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class GridFTPOutputHandler extends AbstractHandler {
    private static final Logger log = LoggerFactory.getLogger(GridFTPOutputHandler.class);
    private Registry registry;


    public void invoke(JobExecutionContext jobExecutionContext) throws GFacHandlerException {
       log.info("Invoking GridFTPOutputHandler ...");
       super.invoke(jobExecutionContext);
        
       ApplicationDeploymentDescriptionType app = jobExecutionContext.getApplicationContext().getApplicationDeploymentDescription().getType();

 	   HostDescriptionType hostType = jobExecutionContext.getApplicationContext().getHostDescription().getType();
 	   String[] gridFTPEndpointArray = null;
 	   String hostName = null;
 
       if(jobExecutionContext.getApplicationContext().getHostDescription().getType() instanceof GlobusHostType){
        	gridFTPEndpointArray = ((GlobusHostType) hostType).getGridFTPEndPointArray();
        	hostName = ((GlobusHostType) hostType).getHostName();
 
       }
       else if (jobExecutionContext.getApplicationContext().getHostDescription().getType() instanceof UnicoreHostType){
        	gridFTPEndpointArray = ((UnicoreHostType) hostType).getGridFTPEndPointArray();
        	hostName = ((UnicoreHostType) hostType).getHostName();
       }
       else {
        	//TODO
       }

       GridFtp ftp = new GridFtp();
       File localStdErrFile = null;
       Map<String, ActualParameter> stringMap = new HashMap<String, ActualParameter>();
	   DataTransferDetails detail = new DataTransferDetails();
	   TransferStatus status = new TransferStatus();

       try {
    	    GSSCredential gssCred = ((GSISecurityContext)jobExecutionContext.getSecurityContext(GSISecurityContext.GSI_SECURITY_CONTEXT)).getGssCredentials();
    	    String[] hostgridFTP = gridFTPEndpointArray;
            if (hostgridFTP == null || hostgridFTP.length == 0) {
                hostgridFTP = new String[]{hostName};
            }
            for (String endpoint : gridFTPEndpointArray) {
                try {
                    /*
                     *  Read Stdout and Stderror
                     */
					URI stdoutURI = GramProviderUtils.createGsiftpURI(endpoint, app.getStandardOutput());
                    URI stderrURI = GramProviderUtils.createGsiftpURI(endpoint, app.getStandardError());
                	status.setTransferState(TransferState.COMPLETE);
					detail.setTransferStatus(status);
					detail.setTransferDescription("STDOUT:" + stdoutURI.toString());
                    registry.add(ChildDataType.DATA_TRANSFER_DETAIL,detail, jobExecutionContext.getTaskData().getTaskID());
                    status.setTransferState(TransferState.COMPLETE);
					detail.setTransferStatus(status);
					detail.setTransferDescription("STDERR:" + stderrURI.toString());
                    registry.add(ChildDataType.DATA_TRANSFER_DETAIL,detail, jobExecutionContext.getTaskData().getTaskID());
                  
                    log.info("STDOUT:" + stdoutURI.toString());
                    log.info("STDERR:" + stderrURI.toString());

                    File logDir = new File("./service_logs");
                    if (!logDir.exists()) {
                        logDir.mkdir();
                    }

                    String timeStampedServiceName = GFacUtils.createUniqueNameForService(jobExecutionContext
                            .getServiceName());
                    File localStdOutFile = File.createTempFile(timeStampedServiceName, "stdout");
                    localStdErrFile = File.createTempFile(timeStampedServiceName, "stderr");


                    String stdout = null;
                    String stderr = null;

                    // TODO: what if job is failed
                    // and this handler is not able to find std* files?
                    try {
                     stdout = ftp.readRemoteFile(stdoutURI, gssCred, localStdOutFile);
                     stderr = ftp.readRemoteFile(stderrURI, gssCred, localStdErrFile);
                     //TODO: do we also need to set them as output parameters for another job
                     ApplicationDescription application = jobExecutionContext.getApplicationContext().getApplicationDeploymentDescription();
                     ApplicationDeploymentDescriptionType appDesc = application.getType();
                     appDesc.setStandardOutput(stdout);
                     appDesc.setStandardError(stderr);
                     jobExecutionContext.getApplicationContext().setApplicationDeploymentDescription(application);
                    }
                    catch(ToolsException e) {
                        log.error("Cannot download stdout/err files. One reason could be the job is not successfully finished:  "+e.getMessage());
                    }

                    List<DataObjectType> outputArray = new ArrayList<DataObjectType>();
                    Map<String, Object> output = jobExecutionContext.getOutMessageContext().getParameters();
                    Set<String> keys = output.keySet();
                    for (String paramName : keys) {
                        ActualParameter actualParameter = (ActualParameter) output.get(paramName);
                        if ("URIArray".equals(actualParameter.getType().getType().toString())) {
                            URI outputURI = GramProviderUtils.createGsiftpURI(endpoint, app.getOutputDataDirectory());
                            List<String> outputList = ftp.listDir(outputURI, gssCred);
                            String[] valueList = outputList.toArray(new String[outputList.size()]);
                            ((URIArrayType) actualParameter.getType()).setValueArray(valueList);
                            stringMap.put(paramName, actualParameter);
                        }else if ("StringArray".equals(actualParameter.getType().getType().toString())) {
                            String[] valueList = OutputUtils.parseStdoutArray(stdout, paramName);
                            ((StringArrayType) actualParameter.getType()).setValueArray(valueList);
                            stringMap.put(paramName, actualParameter);
                        } else if ("URI".equals(actualParameter.getType().getType().toString())) {
                        	  URI outputURI = GramProviderUtils.createGsiftpURI(endpoint, app.getOutputDataDirectory());
                              List<String> outputList = ftp.listDir(outputURI, gssCred);
							if (outputList.size() == 0 || outputList.get(0).isEmpty()) {
								OutputUtils.fillOutputFromStdout(output, stdout, stderr,outputArray);
							} else {
								String valueList = outputList.get(0);
								((URIParameterType) actualParameter.getType()).setValue(valueList);
								stringMap = new HashMap<String, ActualParameter>();
								stringMap.put(paramName, actualParameter);
							}
                        }
                        else {
                            // This is to handle exception during the output parsing.
                            OutputUtils.fillOutputFromStdout(output, stdout, stderr,outputArray);
                        }
                        status.setTransferState(TransferState.DOWNLOAD);
    					detail.setTransferStatus(status);
    					detail.setTransferDescription("Output: " + stringMap.get(paramName).toString());
                        registry.add(ChildDataType.DATA_TRANSFER_DETAIL,detail, jobExecutionContext.getTaskData().getTaskID());
                      
                    }
                    if (outputArray == null || outputArray.isEmpty()) {
                        throw new GFacHandlerException("Empty Output returned from the Application, Double check the application" +
                                "and ApplicationDescriptor output Parameter Names");
                    }
                    // If users has given an output Data path to download the output files this will download the file on machine where GFac is installed
                    TaskDetails taskData =  jobExecutionContext.getTaskData();
                    if(taskData != null && taskData.getAdvancedOutputDataHandling() != null){
                    	String outputDataDirectory = taskData.getAdvancedOutputDataHandling().getOutputDataDir();
                            if(outputDataDirectory != null && !"".equals(outputDataDirectory)){
                                stageOutputFiles(jobExecutionContext,outputDataDirectory);
                            }
                    }
                } catch (ToolsException e) {
                    log.error(e.getMessage());
                    throw new GFacHandlerException(e.getMessage() + "\n StdError Data: \n" +readLastLinesofStdOut(localStdErrFile.getPath(), 20),e);
                } catch (URISyntaxException e) {
                    log.error(e.getMessage());
                    throw new GFacHandlerException("URI is malformatted:" + e.getMessage(), e, readLastLinesofStdOut(localStdErrFile.getPath(), 20));
                }
            }
        } catch (Exception e) {
        	 try {
        	    status.setTransferState(TransferState.FAILED);
				detail.setTransferStatus(status);
				registry.add(ChildDataType.DATA_TRANSFER_DETAIL,detail, jobExecutionContext.getTaskData().getTaskID());
				GFacUtils.saveErrorDetails(jobExecutionContext, e.getLocalizedMessage(), CorrectiveAction.CONTACT_SUPPORT, ErrorCategory.FILE_SYSTEM_FAILURE);
	 		} catch (Exception e1) {
 			    throw new GFacHandlerException("Error persisting status", e1, e1.getLocalizedMessage());
 		   }
        	log.error(e.getMessage());
            throw new GFacHandlerException(e.getMessage(), e, readLastLinesofStdOut(localStdErrFile.getPath(), 20));
        }

    }

    private static String readLastLinesofStdOut(String path, int count) {
        StringBuffer buffer = new StringBuffer();
        FileInputStream in = null;
        try {
            in = new FileInputStream(path);
        } catch (FileNotFoundException e) {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }
        BufferedReader br = new BufferedReader(new InputStreamReader(in));
        List<String> strLine = new ArrayList<String>();
        String tmp = null;
        int numberofLines = 0;
        try {
            while ((tmp = br.readLine()) != null) {
                strLine.add(tmp);
                numberofLines++;
            }
        } catch (IOException e) {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }
        if (numberofLines > count) {
            for (int i = numberofLines - count; i < numberofLines; i++) {
                buffer.append(strLine.get(i));
                buffer.append("\n");
            }
        } else {
            for (int i = 0; i < numberofLines; i++) {
                buffer.append(strLine.get(i));
                buffer.append("\n");
            }
        }
        try {
            in.close();
        } catch (IOException e) {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }
        return buffer.toString();
    }

    private static void stageOutputFiles(JobExecutionContext jobExecutionContext, String outputFileStagingPath) throws GFacProviderException,GFacException, ApplicationSettingsException {


    	   HostDescriptionType hostType = jobExecutionContext.getApplicationContext().getHostDescription().getType();
    	   String[] gridFTPEndpointArray = null;

           if(jobExecutionContext.getApplicationContext().getHostDescription().getType() instanceof GlobusHostType){
           	gridFTPEndpointArray = ((GlobusHostType) hostType).getGridFTPEndPointArray();
           }
           else if (jobExecutionContext.getApplicationContext().getHostDescription().getType() instanceof UnicoreHostType){
           	gridFTPEndpointArray = ((UnicoreHostType) hostType).getGridFTPEndPointArray();
           }
           else {
           	//TODO
           }


        MessageContext outputNew = new MessageContext();
        MessageContext output = jobExecutionContext.getOutMessageContext();
        Map<String, Object> parameters = output.getParameters();
        for (String paramName : parameters.keySet()) {
            ActualParameter actualParameter = (ActualParameter) parameters
                    .get(paramName);

            GridFtp ftp = new GridFtp();
            GSSCredential gssCred = ((GSISecurityContext)jobExecutionContext.getSecurityContext(GSISecurityContext.GSI_SECURITY_CONTEXT)).getGssCredentials();
            try {
                if ("URI".equals(actualParameter.getType().getType().toString())) {
                    for (String endpoint : gridFTPEndpointArray) {
                        ((URIParameterType) actualParameter.getType()).setValue(doStaging(outputFileStagingPath,
                                MappingFactory.toString(actualParameter), ftp, gssCred, endpoint));
                    }
                } else if ("URIArray".equals(actualParameter.getType().getType().toString())) {
                    List<String> split = Arrays.asList(StringUtil.getElementsFromString(MappingFactory.toString(actualParameter)));
                    List<String> newFiles = new ArrayList<String>();
                    for (String endpoint : gridFTPEndpointArray) {
                        for (String paramValueEach : split) {
                            newFiles.add(doStaging(outputFileStagingPath, paramValueEach, ftp, gssCred, endpoint));
                        }
                        ((URIArrayType) actualParameter.getType()).setValueArray(newFiles.toArray(new String[newFiles.size()]));
                    }

                }
            } catch (URISyntaxException e) {
                log.error(e.getMessage());
                throw new GFacProviderException(e.getMessage(), e);
            } catch (ToolsException e) {
                log.error(e.getMessage());
                throw new GFacProviderException(e.getMessage(), e);
            }
            outputNew.getParameters().put(paramName, actualParameter);
        }
        jobExecutionContext.setOutMessageContext(outputNew);
    }

    private static String doStaging(String outputFileStagingPath, String paramValue, GridFtp ftp, GSSCredential gssCred, String endpoint) throws URISyntaxException, ToolsException {
        URI srcURI = GramProviderUtils.createGsiftpURI(endpoint, paramValue);
        String fileName = new File(srcURI.getPath()).getName();
        File outputpath = new File(outputFileStagingPath);
        if(!outputpath.exists()){
        	outputpath.mkdirs();
        }
        File outputFile = new File(outputpath.getAbsolutePath() + File.separator + fileName);
        ftp.readRemoteFile(srcURI,
                gssCred, outputFile);
        return outputFileStagingPath + File.separator + fileName;
    }

    public void initProperties(Properties properties) throws GFacHandlerException {

    }
}
