/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activeio.journal.active;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.NumberFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.activeio.adapter.PacketOutputStream;
import org.apache.activeio.adapter.PacketToInputStream;
import org.apache.activeio.journal.InvalidRecordLocationException;
import org.apache.activeio.journal.active.BatchedWrite;
import org.apache.activeio.journal.active.ControlFile;
import org.apache.activeio.journal.active.Location;
import org.apache.activeio.journal.active.LogFile;
import org.apache.activeio.journal.active.LogFileNode;
import org.apache.activeio.journal.active.Record;
import org.apache.activeio.journal.active.RecordInfo;
import org.apache.activeio.packet.ByteArrayPacket;
import org.apache.activeio.packet.ByteBufferPacket;
import org.apache.activeio.packet.Packet;

public final class LogFileManager {
    public static final int DEFAULT_LOGFILE_COUNT = Integer.parseInt(System.getProperty("org.apache.activeio.journal.active.DefaultLogFileCount", "2"));
    public static final int DEFAULT_LOGFILE_SIZE = Integer.parseInt(System.getProperty("org.apache.activeio.journal.active.DefaultLogFileSize", "20971520"));
    public static final int SERIALIZED_SIZE = 14;
    public static final byte DATA_RECORD_TYPE = 1;
    public static final byte MARK_RECORD_TYPE = 2;
    private static final NumberFormat onlineLogNameFormat = NumberFormat.getNumberInstance();
    private static final NumberFormat archiveLogNameFormat;
    private final File logDirectory;
    private final int initialLogFileSize;
    private final int onlineLogFileCount;
    private final AtomicInteger activeLogFileCount = new AtomicInteger(0);
    private LogFileNode firstNode;
    private LogFileNode firstActiveNode;
    private LogFileNode firstInactiveNode;
    private LogFileNode appendNode;
    private ControlFile controlFile;
    private int lastLogFileId = -1;
    private Location lastMark;
    private boolean disposed;
    private boolean loadedFromCleanShutDown;
    private File archiveDirectory;
    HashMap openArchivedLogs = new HashMap();

    public LogFileManager(File logDirectory) throws IOException {
        this(logDirectory, DEFAULT_LOGFILE_COUNT, DEFAULT_LOGFILE_SIZE, null);
    }

    public LogFileManager(File logDirectory, int onlineLogFileCount, int initialLogFileSize, File archiveDirectory) throws IOException {
        this.logDirectory = logDirectory;
        this.onlineLogFileCount = onlineLogFileCount;
        this.initialLogFileSize = initialLogFileSize;
        this.initialize(onlineLogFileCount);
        this.archiveDirectory = archiveDirectory;
    }

    void initialize(int onlineLogFileCount) throws IOException {
        int i;
        LogFileNode[] logFiles = new LogFileNode[onlineLogFileCount];
        if (!this.logDirectory.exists() && !this.logDirectory.mkdirs()) {
            throw new IOException("Could not create directory: " + this.logDirectory);
        }
        int controlDataSize = 14 + 10 * onlineLogFileCount;
        this.controlFile = new ControlFile(new File(this.logDirectory, "control.dat"), controlDataSize);
        this.controlFile.lock();
        for (i = 0; i < onlineLogFileCount; ++i) {
            LogFile file = new LogFile(new File(this.logDirectory, "log-" + onlineLogNameFormat.format(i) + ".dat"), this.initialLogFileSize);
            logFiles[i] = new LogFileNode(file);
        }
        for (i = 0; i < onlineLogFileCount; ++i) {
            if (i == onlineLogFileCount - 1) {
                logFiles[i].setNext(logFiles[0]);
                continue;
            }
            logFiles[i].setNext(logFiles[i + 1]);
        }
        this.firstNode = logFiles[0];
        this.loadState();
        for (i = 0; i < onlineLogFileCount; ++i) {
            if (!logFiles[i].isActive() || this.firstActiveNode != null && logFiles[i].getId() >= this.firstActiveNode.getId()) continue;
            this.firstActiveNode = logFiles[i];
        }
        if (this.firstActiveNode == null) {
            this.firstInactiveNode = logFiles[0];
            this.activateNextLogFile();
        } else {
            this.firstInactiveNode = null;
            LogFileNode log = this.firstActiveNode;
            do {
                if (!log.isActive()) {
                    this.firstInactiveNode = log;
                    break;
                }
                this.appendNode = log;
            } while ((log = log.getNext()) != this.firstActiveNode);
        }
        if (!this.loadedFromCleanShutDown) {
            this.checkAppendLog();
        }
        this.loadedFromCleanShutDown = false;
        this.storeState();
    }

    private void checkAppendLog() throws IOException {
        int offset;
        Record record = new Record();
        LogFile logFile = this.appendNode.getLogFile();
        Location markLocation = null;
        for (offset = 0; logFile.loadAndCheckRecord(offset, record) && record.getLocation().getLogFileId() == this.appendNode.getId() && record.getLocation().getLogFileOffset() == offset; offset += record.getRecordLength()) {
            if (record.getRecordType() != 2) continue;
            markLocation = record.getLocation();
        }
        this.appendNode.setAppendOffset(offset);
        if (markLocation != null) {
            try {
                Packet packet = this.readPacket(markLocation);
                markLocation = Location.readFromPacket(packet);
            }
            catch (InvalidRecordLocationException e) {
                throw (IOException)new IOException(e.getMessage()).initCause(e);
            }
            this.updateMark(markLocation);
        }
    }

    private void storeState() throws IOException {
        Packet controlData = this.controlFile.getControlData();
        if (controlData.remaining() == 0) {
            return;
        }
        DataOutputStream data = new DataOutputStream(new PacketOutputStream(controlData));
        data.writeInt(this.lastLogFileId);
        data.writeBoolean(this.lastMark != null);
        if (this.lastMark != null) {
            this.lastMark.writeToDataOutput(data);
        }
        data.writeBoolean(this.loadedFromCleanShutDown);
        LogFileNode log = this.firstNode;
        do {
            log.writeExternal(data);
        } while ((log = log.getNext()) != this.firstNode);
        this.controlFile.store();
    }

    private void loadState() throws IOException {
        if (this.controlFile.load()) {
            Packet controlData = this.controlFile.getControlData();
            if (controlData.remaining() == 0) {
                return;
            }
            DataInputStream data = new DataInputStream(new PacketToInputStream(controlData));
            this.lastLogFileId = data.readInt();
            this.lastMark = data.readBoolean() ? Location.readFromDataInput(data) : null;
            this.loadedFromCleanShutDown = data.readBoolean();
            LogFileNode log = this.firstNode;
            do {
                log.readExternal(data);
            } while ((log = log.getNext()) != this.firstNode);
        }
    }

    public void dispose() {
        if (this.disposed) {
            return;
        }
        this.disposed = true;
        try {
            LogFileNode log = this.firstNode;
            do {
                log.getLogFile().dispose();
            } while ((log = log.getNext()) != this.firstNode);
            Iterator iter = this.openArchivedLogs.values().iterator();
            while (iter.hasNext()) {
                ((LogFile)iter.next()).dispose();
            }
            this.loadedFromCleanShutDown = true;
            this.storeState();
            this.controlFile.dispose();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private int getNextLogFileId() {
        return ++this.lastLogFileId;
    }

    public void append(BatchedWrite write) throws IOException {
        if (!this.appendNode.isActive()) {
            throw new IllegalStateException("Log file is not active.  Writes are not allowed");
        }
        if (this.appendNode.isReadOnly()) {
            throw new IllegalStateException("Log file has been marked Read Only.  Writes are not allowed");
        }
        LogFile logFile = this.appendNode.getLogFile();
        ByteBuffer buffer = ((ByteBufferPacket)write.getPacket().getAdapter(ByteBufferPacket.class)).getByteBuffer();
        int size = buffer.remaining();
        logFile.write(this.appendNode.getAppendOffset(), buffer);
        if (write.getForce()) {
            logFile.force();
        }
        this.appendNode.appended(size);
        if (write.getMark() != null) {
            this.updateMark(write.getMark());
        }
    }

    private synchronized void updateMark(Location mark) throws IOException {
        this.lastMark = mark;
        while (this.firstActiveNode != this.appendNode && this.firstActiveNode.getId() < this.lastMark.getLogFileId()) {
            if (this.archiveDirectory != null) {
                File file = this.getArchiveFile(this.firstActiveNode.getId());
                this.firstActiveNode.getLogFile().copyTo(file);
            }
            this.firstActiveNode.deactivate();
            this.activeLogFileCount.decrementAndGet();
            if (this.firstInactiveNode == null) {
                this.firstInactiveNode = this.firstActiveNode;
            }
            this.firstActiveNode = this.firstActiveNode.getNextActive();
        }
    }

    private File getArchiveFile(int logId) {
        return new File(this.archiveDirectory, "" + archiveLogNameFormat.format(logId) + ".log");
    }

    RecordInfo readRecordInfo(Location location) throws IOException, InvalidRecordLocationException {
        LogFile logFile;
        LogFileNode logFileState = this.getLogFileWithId(location.getLogFileId());
        if (logFileState != null) {
            if (logFileState.getAppendOffset() == location.getLogFileOffset()) {
                throw new InvalidRecordLocationException("No record at (" + location + ") found.  Location past end of logged data.");
            }
            logFile = logFileState.getLogFile();
        } else {
            if (this.archiveDirectory == null) {
                throw new InvalidRecordLocationException("Log file: " + location.getLogFileId() + " is not active.");
            }
            logFile = this.getArchivedLogFile(location.getLogFileId());
        }
        try {
            Record header = new Record();
            logFile.readRecordHeader(location.getLogFileOffset(), header);
            return new RecordInfo(location, header, logFileState, logFile);
        }
        catch (IOException e) {
            throw new InvalidRecordLocationException("No record at (" + location + ") found.");
        }
    }

    private LogFile getArchivedLogFile(int logFileId) throws InvalidRecordLocationException, IOException {
        Integer key = new Integer(logFileId);
        LogFile rc = (LogFile)this.openArchivedLogs.get(key);
        if (rc == null) {
            File archiveFile = this.getArchiveFile(logFileId);
            if (!archiveFile.canRead()) {
                throw new InvalidRecordLocationException("Log file: " + logFileId + " does not exist.");
            }
            rc = new LogFile(archiveFile, this.getInitialLogFileSize());
            this.openArchivedLogs.put(key, rc);
        }
        return rc;
    }

    LogFileNode getLogFileWithId(int logFileId) throws InvalidRecordLocationException {
        for (LogFileNode lf = this.firstActiveNode; lf != null; lf = lf.getNextActive()) {
            if (lf.getId() == logFileId) {
                return lf;
            }
            if (logFileId < lf.getId()) break;
        }
        return null;
    }

    public Location getNextDataRecordLocation(Location lastLocation) throws IOException, InvalidRecordLocationException {
        RecordInfo ri = this.readRecordInfo(lastLocation);
        do {
            int logFileId = ri.getLocation().getLogFileId();
            int offset = ri.getNextLocation();
            if (offset >= ri.getLogFileState().getAppendOffset()) {
                LogFileNode nextActive = ri.getLogFileState().getNextActive();
                if (nextActive == null || nextActive.getId() <= ri.getLogFileState().getId()) {
                    return null;
                }
                logFileId = nextActive.getId();
                offset = 0;
            }
            try {
                ri = this.readRecordInfo(new Location(logFileId, offset));
            }
            catch (InvalidRecordLocationException e) {
                return null;
            }
        } while (ri.getHeader().getRecordType() != 1);
        return ri.getLocation();
    }

    public Packet readPacket(Location location) throws IOException, InvalidRecordLocationException {
        RecordInfo recordInfo = this.readRecordInfo(location);
        byte[] data = new byte[recordInfo.getHeader().getPayloadLength()];
        LogFile logFile = recordInfo.getLogFile();
        logFile.read(recordInfo.getDataOffset(), data);
        return new ByteArrayPacket(data);
    }

    public int getInitialLogFileSize() {
        return this.initialLogFileSize;
    }

    public Location getFirstActiveLogLocation() {
        if (this.firstActiveNode == null) {
            return null;
        }
        if (this.firstActiveNode.getAppendOffset() == 0) {
            return null;
        }
        return new Location(this.firstActiveNode.getId(), 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void activateNextLogFile() throws IOException {
        if (this.appendNode != null) {
            this.appendNode.setReadOnly(true);
        }
        LogFileNode next = this.firstInactiveNode;
        LogFileManager logFileManager = this;
        synchronized (logFileManager) {
            this.firstInactiveNode = this.firstInactiveNode.getNextInactive();
            next.activate(this.getNextLogFileId());
            if (this.firstActiveNode == null) {
                this.firstActiveNode = next;
            }
        }
        this.activeLogFileCount.incrementAndGet();
        this.appendNode = next;
        this.storeState();
    }

    public File getLogDirectory() {
        return this.logDirectory;
    }

    public Location getLastMarkedRecordLocation() {
        return this.lastMark;
    }

    public Location getNextAppendLocation() {
        return new Location(this.appendNode.getId(), this.appendNode.getAppendOffset());
    }

    public int getOnlineLogFileCount() {
        return this.onlineLogFileCount;
    }

    public boolean isPastHalfActive() {
        return (float)this.onlineLogFileCount / 2.0f < (float)this.activeLogFileCount.get();
    }

    public synchronized Location getFirstRecordLocationOfSecondActiveLogFile() {
        return this.firstActiveNode.getNextActive().getFirstRecordLocation();
    }

    public synchronized boolean canActivateNextLogFile() {
        return this.firstInactiveNode != null;
    }

    static {
        onlineLogNameFormat.setMinimumIntegerDigits(3);
        onlineLogNameFormat.setMaximumIntegerDigits(3);
        onlineLogNameFormat.setGroupingUsed(false);
        onlineLogNameFormat.setParseIntegerOnly(true);
        onlineLogNameFormat.setMaximumFractionDigits(0);
        archiveLogNameFormat = NumberFormat.getNumberInstance();
        archiveLogNameFormat.setMinimumIntegerDigits(8);
        archiveLogNameFormat.setMaximumIntegerDigits(8);
        archiveLogNameFormat.setGroupingUsed(false);
        archiveLogNameFormat.setParseIntegerOnly(true);
        archiveLogNameFormat.setMaximumFractionDigits(0);
    }
}

