/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.logs;

import android.support.annotation.NonNull;
import org.amoustakos.bugreporter.utils.ErrorMessages;
import org.amoustakos.bugreporter.utils.BugReporterUtils;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;


/**
 * Class to provide basic logcat reading with filtering by PID and log level.
 */
public class LogUtil {
    private static final String TAG = ErrorMessages.BASE_NAME + LogUtil.class.getSimpleName();
    private static final int PID = android.os.Process.myPid();

    /**
     * Records all logs (without filter) for this session.
     */
    @NonNull public static StringBuilder readAllLogs(@NonNull String level) {
        StringBuilder logBuilder = new StringBuilder();
        try {
            String[] command = new String[] {"logcat", level};
            Process process = Runtime.getRuntime().exec(command);
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                logBuilder.append(line);
                logBuilder.append("\n");
            }
        } catch (IOException e) {
            BugReporterUtils.logW(TAG, e.getMessage(), e);
        }
        return logBuilder;
    }

    /**
     * Records the application's logs for this session.
     */
    @NonNull public static StringBuilder readApplicationLogs(@NonNull String level) {
        StringBuilder logBuilder = new StringBuilder();
        try {
            String[] command = new String[] {"logcat", level};
            Process process = Runtime.getRuntime().exec(command);
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            while ((line = bufferedReader.readLine()) != null) {
                if(line.contains(PID+"")) {
                    logBuilder.append(line);
                    logBuilder.append("\n");
                }
            }
        } catch (IOException e) {
            BugReporterUtils.logW(TAG, e.getMessage(), e);
        }
        return logBuilder;
    }

}
