/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.collectors.bugs;

import android.support.annotation.Nullable;
import org.amoustakos.bugreporter.utils.ErrorMessages;

/**
 * This class represents basic Bug information for bug reporting:<br>
 * 1. Message<br>
 * 2. Culprit class<br>
 *<br>
 * This is an abstract class that needs to be extended.<br>
 * One of the included extending classes can be included for bug gathering-reporting.<br>
 * See: {@link BugWithLog}, {@link BugWithThrowable}
 */
public abstract class Bug {
    public static final String TAG = ErrorMessages.BASE_NAME + Bug.class.getSimpleName();

    @Nullable private String message;
    @Nullable private Class culprit;

    /*
     * Constructors
     */
    /**
     * Empty constructor to initialize variables.
     */
    public Bug() {
        this.message = null;
        this.culprit = null;
    }
    /**
     * Constructor with all available variables.
     * @param message A text that describes the bug (e.g. a throwable error message)
     * @param culprit The class responsible for the error.
     */
    public Bug(@Nullable String message, @Nullable Class culprit) {
        this.message = message;
        this.culprit = culprit;
    }

    /*
     * Methods
     */
    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(){
        return buildTextReport(new StringBuilder());
    }

    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(StringBuilder report){
        report.append("|-------------------------------------------|\n");
        report.append("|                BUG Information            |\n");
        report.append("|-------------------------------------------|\n");
        report.append("Culprit: "); report.append(getCulprit()==null?"":getCulprit().getName()); report.append("\n");
        report.append("Message: "); report.append("\n"); report.append(getMessage()); report.append("\n");
        report.append("|                                           |\n");
        report.append("|                                           |\n");
        report.append("|-------------------------------------------|\n");
        return report;
    }

    /*
     * Getters - Setters
     */
    /**
     * A text that describes the bug (e.g. a throwable error message).
     */
    @Nullable public String getMessage() {return message;}
    /**
     * Sets the error text that describes the bug (e.g. a throwable error message).
     */
    public synchronized Bug setMessage(@Nullable String message) {this.message = message; return this;}
    /**
     * The class responsible for the error.
     */
    @Nullable public Class getCulprit() {return culprit;}
    /**
     * Sets the class responsible for the error.
     */
    public synchronized Bug setCulprit(@Nullable Class culprit) {this.culprit = culprit; return this;}
}
