/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.builders;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import org.amoustakos.bugreporter.collectors.application.ApplicationInfo;
import org.amoustakos.bugreporter.collectors.bugs.Bug;
import org.amoustakos.bugreporter.collectors.bugs.BugWithThrowable;
import org.amoustakos.bugreporter.collectors.device.DeviceInfo;
import org.amoustakos.bugreporter.collectors.device.OSInfo;
import org.amoustakos.bugreporter.utils.ErrorMessages;
import org.amoustakos.bugreporter.utils.BugReporterUtils;

/**
 * Default {@link BugBuilder} class for registering a {@link BugWithThrowable}.<br>
 *
 * @see BugBuilder
 */
public class ThrowableBugBuilder extends BugBuilder {
    /*
     * Constructors
     */
    /**
     * Default constructor.
     * @see BugBuilder#BugBuilder(DeviceInfo, OSInfo, ApplicationInfo, Bug)
     */
    public ThrowableBugBuilder(@NonNull DeviceInfo device,
                               @NonNull OSInfo os,
                               @NonNull ApplicationInfo application,
                               @NonNull BugWithThrowable bug) {
        super(device, os, application, bug);
    }

    /*
     * Dev Info
     */
    /**
     * Gathers default {@link DeviceInfo}.
     */
    public static DeviceInfo gatherDeviceInfo(@NonNull Context context) {
        return DeviceInfo.getAutoDevInfo(context);
    }

    /*
     * OS Info
     */
    /**
     * Gathers default {@link OSInfo}.
     */
    public static OSInfo gatherOSInfo() {return OSInfo.getAutoOSInfo();}

    /*
     * Application Info
     */
    /**
     * Gathers default {@link ApplicationInfo}.
     */
    public static ApplicationInfo gatherAppInfo(@Nullable String name,
                                                @Nullable String packageName,
                                                @Nullable String versionName,
                                                int versionCode){
        return new ApplicationInfo(name, packageName, versionName, versionCode);
    }

    /*
     * Bug info
     */
    /**
     * Gathers bug information with the {@link Throwable} provided.
     */
    public static BugWithThrowable gatherBugInfo(@NonNull Throwable t) {return new BugWithThrowable(t);}
    /**
     * Gathers bug information with the {@link Throwable} provided, a message to describe the bug <br>
     * and the responsible {@link Class}.
     */
    public static BugWithThrowable gatherBugInfo(@NonNull Throwable t, @NonNull Class culprit) {
        return new BugWithThrowable(t, culprit);
    }

    /*
     * Builders
     */
    /**
     * Builds bug information.
     * @param context [REQUIRED] Application context.
     * @param t [REQUIRED] The {@link Throwable} associated with this bug.
     * @param appName [OPTIONAL] The application's name.
     * @param packageName [OPTIONAL] The application's package name.
     * @param versionName [OPTIONAL] The application version name.
     * @param versionCode [OPTIONAL] The application version code.
     */
    public static ThrowableBugBuilder build(@NonNull Context context,
                                            @NonNull Throwable t,
                                            @Nullable String appName,
                                            @Nullable String packageName,
                                            @Nullable String versionName,
                                            int versionCode) {
        if(!BugReporterUtils.isLibraryInitialized(TAG) || BugReporterUtils.isContextNull(TAG, context))
            return null;

        DeviceInfo dev = gatherDeviceInfo(context);
        if(dev == null){
            BugReporterUtils.logE(TAG, ErrorMessages.DEV_INFO_COULD_NOT_GATHER);
            return null;
        }
        OSInfo osInfo = gatherOSInfo();
        ApplicationInfo appInfo = gatherAppInfo(appName, packageName, versionName, versionCode);
        BugWithThrowable bug = gatherBugInfo(t);
        return new ThrowableBugBuilder(dev, osInfo, appInfo, bug);
    }
    /**
     * Builds bug information.
     * @param context [REQUIRED] Application context.
     * @param t [REQUIRED] The {@link Throwable} associated with this bug.
     * @param appName [OPTIONAL] The application's name.
     * @param packageName [OPTIONAL] The application's package name.
     * @param versionName [OPTIONAL] The application version name.
     * @param versionCode [OPTIONAL] The application version code.
     * @param culprit [OPTIONAL] The responsible {@link Class}.
     */
    public static ThrowableBugBuilder build(@NonNull Context context,
                                            @NonNull Throwable t,
                                            @Nullable String appName,
                                            @Nullable String packageName,
                                            @Nullable String versionName,
                                            int versionCode,
                                            @NonNull Class culprit) {
        if(!BugReporterUtils.isLibraryInitialized(TAG) || BugReporterUtils.isContextNull(TAG, context))
            return null;

        DeviceInfo dev = gatherDeviceInfo(context);
        if(dev == null){
            BugReporterUtils.logE(TAG, ErrorMessages.DEV_INFO_COULD_NOT_GATHER);
            return null;
        }
        OSInfo osInfo = gatherOSInfo();
        ApplicationInfo appInfo = gatherAppInfo(appName, packageName, versionName, versionCode);
        BugWithThrowable bug = gatherBugInfo(t, culprit);
        return new ThrowableBugBuilder(dev, osInfo, appInfo, bug);
    }
}
