/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.builders;

import android.support.annotation.NonNull;
import org.amoustakos.bugreporter.collectors.application.ApplicationInfo;
import org.amoustakos.bugreporter.collectors.bugs.Bug;
import org.amoustakos.bugreporter.collectors.device.DeviceInfo;
import org.amoustakos.bugreporter.collectors.device.OSInfo;
import org.amoustakos.bugreporter.utils.ErrorMessages;

/**
 * This is an abstract class that should be extended by all builders.<br>
 * It holds the basic information for all bugs:<br>
 * 1. {@link DeviceInfo}<br>
 * 2. {@link OSInfo}<br>
 * 3. {@link ApplicationInfo}<br>
 * 4. {@link Bug}<br>
 *<br>
 * One of the included extending classes can be included for bug gathering-reporting.<br>
 * See: {@link LogBugBuilder}, {@link ThrowableBugBuilder}
 */
public abstract class BugBuilder {
    public static final String TAG = ErrorMessages.BASE_NAME + BugBuilder.class.getSimpleName();

    private DeviceInfo device;
    private OSInfo os;
    private ApplicationInfo application;
    private Bug bug;


    /*
     * Constructors
     */
    public BugBuilder(@NonNull DeviceInfo device, @NonNull OSInfo os, @NonNull ApplicationInfo application, @NonNull Bug bug) {
        this.device = device;
        this.os = os;
        this.application = application;
        this.bug = bug;
    }
    public BugBuilder() {}


    /*
     * getters - Setters
     */
    /**
     * Gets the {@link DeviceInfo} associated with this bug.
     */
    @NonNull public DeviceInfo getDeviceInfo() {return device;}
    /**
     * Sets the {@link DeviceInfo} associated with this bug.
     */
    public synchronized BugBuilder setDeviceInfo(@NonNull DeviceInfo device) {this.device = device; return this;}
    /**
     * Gets the {@link Bug} associated with this bug.
     */
    @NonNull public Bug getBugInfo() {return bug;}
    /**
     * Sets the {@link Bug} associated with this bug.
     */
    public synchronized BugBuilder setBugInfo(@NonNull Bug bug) {this.bug = bug; return this;}
    /**
     * Gets the {@link OSInfo} associated with this bug.
     */
    @NonNull public OSInfo getOSInfo() {return os;}
    /**
     * Sets the {@link OSInfo} associated with this bug.
     */
    public synchronized BugBuilder setOSInfo(@NonNull OSInfo os) {this.os = os; return this;}
    /**
     * Gets the {@link ApplicationInfo} associated with this bug.
     */
    @NonNull public ApplicationInfo getAppInfo() {return application;}
    /**
     * Sets the {@link ApplicationInfo} associated with this bug.
     */
    public synchronized BugBuilder setAppInfo(@NonNull ApplicationInfo application) {this.application = application; return this;}
}
