/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.collectors.device;

import android.annotation.TargetApi;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * This class represents basic OS information for bug reporting:<br>
 * 1. Codename<br>
 * 2. Incremental (build hash/number)<br>
 * 3. SDK Version<br>
 * 4. Preview Release<br>
 *<br>
 * You are free to choose the information to include in this class but it is<br>
 * recommended to have at least these 4 fields.
 */
public class OSInfo {
    @Nullable private String codename;
    @Nullable private String incremental;
    private int sdkVersion;

    @Nullable private Boolean isPreviewRelease;


    /*
     * Constructors
     */
    /**
     * Empty constructor to instantiate all variables as null.<br>
     * (sdk version as -1).
     */
    public OSInfo() {
        this.codename = null;
        this.incremental = null;
        this.sdkVersion = -1;
        this.isPreviewRelease = null;
    }
    /**
     * Constructor with all variables.
     * @param codename The android version codename.
     * @param incremental The OS build hash/number.
     * @param sdkVersion The OS sdk version (e.g. 23).
     * @param isPreviewRelease Whether this OS is a preview release (API 23 +).
     */
    public OSInfo(@Nullable String codename,
                  @Nullable String incremental,
                  int sdkVersion,
                  @Nullable Boolean isPreviewRelease) {
        this.codename = codename;
        this.incremental = incremental;
        this.sdkVersion = sdkVersion;
        this.isPreviewRelease = isPreviewRelease;
    }

    /*
     * Methods
     */
    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(){
        return buildTextReport(new StringBuilder());
    }

    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(StringBuilder report){
        report.append("|-------------------------------------------|\n");
        report.append("|                OS Information             |\n");
        report.append("|-------------------------------------------|\n");
        report.append("Codename: "); report.append(getCodename()==null?"":getCodename()); report.append("\n");
        report.append("SDK version: "); report.append(getSdkVersion()); report.append("\n");
        report.append("Incremental version: "); report.append(getIncremental()); report.append("\n");
        report.append("|                                           |\n");
        report.append("|                                           |\n");
        report.append("|-------------------------------------------|\n");
        return report;
    }


    /*
     * Getters - Setters
     */
    /**
     * The OS codename (e.g. Marshmallow)
     */
    @Nullable public String getCodename() {return codename;}
    /**
     * Sets the OS codename
     */
    public synchronized OSInfo setCodename(@Nullable String codename) {this.codename = codename; return this;}
    /**
     * The OS build hash/number
     */
    @Nullable public String getIncremental() {return incremental;}
    /**
     * Sets the OS build hash/number
     */
    public synchronized OSInfo setIncremental(@Nullable String incremental) {this.incremental = incremental; return this;}
    /**
     * Get the OS SDK version number (e.g. 23)
     */
    @NonNull public int getSdkVersion() {return sdkVersion;}
    /**
     * Sets the OS SDK version number (e.g. 23)
     */
    public synchronized OSInfo setSdkVersion(int sdkVersion) {this.sdkVersion = sdkVersion; return this;}
    /**
     * Boolean representing whether this OS is a preview build
     * @since API 23
     */
    @TargetApi(23) @Nullable public boolean isPreviewRelease() {return isPreviewRelease;}
    /**
     * Sets whether the OS is a review build
     * @since API 23
     */
    @TargetApi(23) public synchronized OSInfo setPreviewRelease(@Nullable Boolean isPreviewRelease) {this.isPreviewRelease = isPreviewRelease; return this;}


    /*
     * Auto-gen
     */
    /**
     * Auto generates an instance of {@link OSInfo} with all variables filled with <br>
     * the default values using the {@link Build} class.
     */
    @NonNull public static OSInfo getAutoOSInfo(){
        return new OSInfo()
                    .setCodename(Build.VERSION.CODENAME)
                    .setIncremental(Build.VERSION.INCREMENTAL)
                    .setPreviewRelease((Build.VERSION.SDK_INT>=Build.VERSION_CODES.M)?(Build.VERSION.PREVIEW_SDK_INT!=0):null)
                    .setSdkVersion(Build.VERSION.SDK_INT);
    }
}
