/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.collectors.device;

import android.content.Context;
import android.content.res.Resources;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.DisplayMetrics;
import org.amoustakos.bugreporter.R;
import org.amoustakos.bugreporter.utils.ErrorMessages;
import org.amoustakos.bugreporter.utils.Utils;

/**
 * This class represents basic Device information for bug reporting:<br>
 * 1. Make<br>
 * 2. Model<br>
 * 3. Board<br>
 * 4. Bootloader<br>
 * 5. Supported ABIs<br>
 *<br>
 * You are free to choose the information to include in this class but it is<br>
 * recommended to have at least these 5 fields.
 */
public class DeviceInfo {
    private static final String TAG = ErrorMessages.BASE_NAME + DeviceInfo.class.getSimpleName();

    @Nullable private String make;
    @Nullable private String model;
    @Nullable private String board;
    @Nullable private String bootloader;
    @Nullable private String[] supportedABIs;
    @Nullable private String dpi;

    /*
     * Constructors
     */
    /**
     * Empty constructor.<br>
     * Instantiates all variables as null.
     */
    public DeviceInfo() {
        this.make = null;
        this.model = null;
        this.board = null;
        this.bootloader = null;
        this.supportedABIs = null;
        this.dpi = null;
    }
    /**
     * Constructor with all variables.
     * @param make The manufacturer of this device
     * @param model The device model
     * @param board The board name (e.g. "hammerhead" for Nexus 5)
     * @param bootloader The bootloader version number for this device
     * @param supportedABIs A list of all ABIs supported by the device
     */
    public DeviceInfo(@Nullable String make,
                      @Nullable String model,
                      @Nullable String board,
                      @Nullable String bootloader,
                      @Nullable String[] supportedABIs,
                      @Nullable String dpi) {
        this.make = make;
        this.model = model;
        this.board = board;
        this.bootloader = bootloader;
        this.supportedABIs = supportedABIs;
        this.dpi = dpi;
    }

    /*
     * Methods
     */
    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(){
        return buildTextReport(new StringBuilder());
    }

    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(StringBuilder report){
        report.append("|-------------------------------------------|\n");
        report.append("|              Device Information           |\n");
        report.append("|-------------------------------------------|\n");
        report.append("Make: "); report.append(getMake()==null?"":getMake()); report.append("\n");
        report.append("Model: "); report.append(getModel()==null?"":getModel()); report.append("\n");
        report.append("Board: "); report.append(getBoard()==null?"":getBoard()); report.append("\n");
        report.append("Bootloader: "); report.append(getBootloader()==null?"":getBootloader()); report.append("\n");
        report.append("Supported ABIs: ");
        if(getSupportedABIs() == null)
            report.append("\n");
        else {
            report.append("\n");
            for (String abi : getSupportedABIs()) {
                report.append("\t");
                report.append(abi);
            }
            report.append("\n");
        }
        report.append("Screen density: "); report.append(getDpi()==null?"":getDpi()); report.append("\n");
        report.append("|                                           |\n");
        report.append("|                                           |\n");
        report.append("|-------------------------------------------|\n");
        return report;
    }


    /*
     * Getters - Setters
     */
    /**
     * The manufacturer of this device
     */
    @Nullable public String getMake() {return make;}
    /**
     * Sets the manufacturer of the device
     */
    public synchronized DeviceInfo setMake(@Nullable String make) {this.make = make; return this;}
    /**
     * The device model
     */
    @Nullable public String getModel() {return model;}
    /**
     * Sets the device model
     */
    public synchronized DeviceInfo setModel(@Nullable String model) {this.model = model; return this;}
    /**
     * The board name (e.g. "hammerhead" for Nexus 5)
     */
    @Nullable public String getBoard() {return board;}
    /**
     * Sets the board name (e.g. "hammerhead" for Nexus 5)
     */
    public synchronized DeviceInfo setBoard(@Nullable String board) {this.board = board; return this;}
    /**
     * The bootloader version number
     */
    @Nullable public String getBootloader() {return bootloader;}
    /**
     * Sets the bootloader version number
     */
    public synchronized DeviceInfo setBootloader(@Nullable String bootloader) {this.bootloader = bootloader; return this;}
    /**
     * A list of all ABIs supported by the device
     */
    @Nullable public String[] getSupportedABIs() {return supportedABIs;}
    /**
     * Sets the ABIs supported by the device
     */
    public synchronized DeviceInfo setSupportedABIs(@Nullable String[] supportedABIs) {this.supportedABIs = supportedABIs; return this;}
    /**
     * Gets the dpi of the device's screen.
     */
    @Nullable public String getDpi() {return dpi;}
    /**
     * Sets the dpi of the device's screen.
     */
    public synchronized DeviceInfo setDpi(@Nullable String dpi) {this.dpi = dpi; return this;}

    /*
     * Auto-gen
     */
    /**
     * Auto generates an instance of {@link DeviceInfo} with all variables filled with <br>
     * the default values using the {@link Build} class.
     */
    @Nullable public static DeviceInfo getAutoDevInfo(@NonNull Context context){
        if(!Utils.isLibraryInitialized(TAG) || Utils.isContextNull(TAG, context))
            return null;

        DisplayMetrics displayMetrics = Resources.getSystem().getDisplayMetrics();
        String dpi = "";
        switch(displayMetrics.densityDpi){
            case DisplayMetrics.DENSITY_LOW:
                dpi = context.getString(R.string.dpi_low);
            break;
            case DisplayMetrics.DENSITY_MEDIUM:
                dpi = context.getString(R.string.dpi_medium);
            break;
            case DisplayMetrics.DENSITY_HIGH:
                dpi = context.getString(R.string.dpi_high);
            break;
            case DisplayMetrics.DENSITY_XHIGH:
                dpi = context.getString(R.string.dpi_xhigh);
            break;
            case DisplayMetrics.DENSITY_XXHIGH:
                dpi = context.getString(R.string.dpi_xxhigh);
            break;
            case DisplayMetrics.DENSITY_XXXHIGH:
                dpi = context.getString(R.string.dpi_xxxhigh);
            break;
        }

        return new DeviceInfo()
                    .setBoard(Build.BOARD)
                    .setBootloader(Build.BOOTLOADER)
                    .setDpi(dpi)
                    .setMake(Build.MANUFACTURER)
                    .setModel(Build.MODEL)
                    .setSupportedABIs(Build.SUPPORTED_ABIS);
    }
}
