/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.collectors.bugs;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import org.amoustakos.bugreporter.utils.ErrorMessages;
import org.amoustakos.bugreporter.utils.Utils;

/**
 * Default {@link Bug} with the addition of a {@link Throwable} element<br>
 * to hold associated error.<br>
 *
 * @see Bug
 */
public class BugWithThrowable extends Bug{
    @Nullable private Throwable throwable;

    /*
     * Constructors
     */
    /**
     * @see Bug#Bug()
     */
    public BugWithThrowable() {
        super();
        setThrowable(null);
    }
    /**
     * Default {@link Bug} constructor with throwable input.<br>
     * @see Bug#Bug()
     */
    public BugWithThrowable(@NonNull Throwable throwable) {
        super();
        setThrowable(throwable);
    }
    /**
     * Default {@link Bug} constructor with throwable and responsible class input.<br>
     * @see Bug#Bug()
     */
    public BugWithThrowable(@Nullable Throwable throwable, @Nullable Class culprit) {
        super();
        setCulprit(culprit);
        setThrowable(throwable);
    }

    private void buildMessage(){
        if(getThrowable() == null) {
            setMessage("");
            Utils.logW(TAG, ErrorMessages.THROWABLE_NULL, new Exception(ErrorMessages.THROWABLE_NULL));
            return;
        }
        StringBuilder message = new StringBuilder();
        for(StackTraceElement el : getThrowable().getStackTrace()){
            message.append(el.toString());
            message.append("\n");
        }
        setMessage(message.toString());
    }


    /*
     * Getters - Setters
     */
    /**
     * Gets the {@link Throwable} element associated with this bug.
     */
    @Nullable public Throwable getThrowable() {return throwable;}
    /**
     * Sets the {@link Throwable} element associated with this bug.
     */
    public synchronized void setThrowable(@Nullable Throwable throwable) {
        this.throwable = throwable;
        buildMessage();
    }
}
