/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.collectors.application;

import android.support.annotation.Nullable;

/**
 * This class represents basic application information for bug reporting:<br>
 * 1. Name<br>
 * 2. Package name<br>
 * 3. Version name<br>
 * 4. Version code<br>
 *<br>
 * You are free to choose the information to include in this class but it is<br>
 * recommended to have at least these 3 fields.
 */
public class ApplicationInfo {
    @Nullable private String name;
    @Nullable private String packageName;
    @Nullable private String versionName;
    private int versionCode;

    /*
     * Constructors
     */
    /**
     * Empty constructor.<br>
     * All internal values are instantiated as null.
     */
    public ApplicationInfo() {
        this.name = null;
        this.packageName = null;
        this.versionName = null;
        this.versionCode = -1;
    }
    /**
     * Constructor with all variables.
     *
     * @param name The application name (nullable).
     * @param packageName The application package name (nullable).
     * @param versionName The application versionName (nullable).
     */
    public ApplicationInfo(@Nullable String name,
                           @Nullable String packageName,
                           @Nullable String versionName,
                           int versionCode) {
        this.name = name;
        this.packageName = packageName;
        this.versionName = versionName;
        this.versionCode = versionCode;
    }
    
    /*
     * Methods
     */
    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(){
        return buildTextReport(new StringBuilder());
    }

    /**
     * Builds a text report with the gathered information
     */
    public StringBuilder buildTextReport(StringBuilder report){
        report.append("|-------------------------------------------|\n");
        report.append("|                APP Information            |\n");
        report.append("|-------------------------------------------|\n");
        report.append("Name: "); report.append(getName()); report.append("\n");
        report.append("Package: "); report.append(getPackageName()); report.append("\n");
        report.append("Version Name: "); report.append(getVersionName()); report.append("\n");
        report.append("Version Code: "); report.append(getVersionCode()); report.append("\n");
        report.append("|                                           |\n");
        report.append("|                                           |\n");
        report.append("|-------------------------------------------|\n");
        return report;
    }


    /*
     * Getters - Setters
     */
    /**
     * The application name.
     */
    @Nullable public String getName() {return name;}
    /**
     * Sets the application name.
     */
    public synchronized ApplicationInfo setName(@Nullable String name) {this.name = name; return this;}
    /**
     * The package name of the application.
     */
    @Nullable public String getPackageName() {return packageName;}
    /**
     * Sets the package name of the application.
     */
    public synchronized ApplicationInfo setPackageName(@Nullable String packageName) {this.packageName = packageName; return this;}
    /**
     * The application version name.
     */
    @Nullable public String getVersionName() {return versionName;}
    /**
     * Sets the application version name
     */
    public synchronized ApplicationInfo setVersionName(@Nullable String versionName) {this.versionName = versionName; return this;}
    /**
     * The application version code.
     */
    public int getVersionCode() {return versionCode;}
    /**
     * Sets the application version code
     */
    public synchronized ApplicationInfo setVersionCode(int versionCode) {this.versionCode = versionCode; return this;}
}
