/**
 *   BugReporter Android library
 *
 *   @author Antonis Moustakos
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License version 2,
 *   as published by the Free Software Foundation.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.amoustakos.bugreporter.builders;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import org.amoustakos.bugreporter.collectors.application.ApplicationInfo;
import org.amoustakos.bugreporter.collectors.bugs.Bug;
import org.amoustakos.bugreporter.collectors.bugs.BugWithLog;
import org.amoustakos.bugreporter.collectors.device.DeviceInfo;
import org.amoustakos.bugreporter.collectors.device.OSInfo;
import org.amoustakos.bugreporter.logs.LogUtil;
import org.amoustakos.bugreporter.utils.ErrorMessages;
import org.amoustakos.bugreporter.utils.Utils;

/**
 * Default {@link BugBuilder} class for registering a {@link BugWithLog}.<br>
 *
 * @see BugBuilder
 */
public class LogBugBuilder extends BugBuilder{
    private String logLevel = null;
    /*
     * App info
     */
    private String appName = null;
    private String packageName = null;
    private String versionName = null;
    private int versionCode = -1;

    private String optionalMessage = null;
    private Class culprit = null;

    /*
     * Constructors
     */
    /**
     * Empty constructor.
     */
    public LogBugBuilder() {
        super();
    }

    /**
     * Default constructor.
     * @see BugBuilder#BugBuilder(DeviceInfo, OSInfo, ApplicationInfo, Bug)
     */
    public LogBugBuilder(@NonNull DeviceInfo device,
                         @NonNull OSInfo os,
                         @NonNull ApplicationInfo application,
                         @NonNull BugWithLog bug) {
        super(device, os, application, bug);
    }

    /*
     * Dev Info
     */
    /**
     * Gathers default {@link DeviceInfo}.
     */
    public static DeviceInfo gatherDeviceInfo(Context context){return DeviceInfo.getAutoDevInfo(context);}

    /*
     * OS Info
     */
    /**
     * Gathers default {@link OSInfo}.
     */
    public static OSInfo gatherOSInfo(){return OSInfo.getAutoOSInfo();}

    /*
     * Application Info
     */
    /**
     * Gathers default {@link ApplicationInfo}.
     */
    public static ApplicationInfo gatherAppInfo(@Nullable String name,
                                                @Nullable String packageName,
                                                @Nullable String versionName,
                                                int versionCode){
        return new ApplicationInfo(name, packageName, versionName, versionCode);
    }

    /*
     * Bug info
     */
    /**
     * Gathers bug information by recording the application's logs for this session.
     */
    public static BugWithLog gatherBugInfo(@NonNull String logLevel){
        return new BugWithLog(LogUtil.readApplicationLogs(logLevel));
    }
    /**
     * Gathers bug information by recording the application's logs for this session and setting an optional message.
     */
    public static BugWithLog gatherBugInfo(@NonNull String logLevel, @NonNull String optionalMessage){
        return new BugWithLog(optionalMessage, LogUtil.readApplicationLogs(logLevel));
    }
    /**
     * Gathers bug information by recording the application's logs for this session and optionally setting <br>
     * a message and the {@link Class} responsible.
     */
    public static BugWithLog gatherBugInfo(@NonNull String logLevel, @Nullable String optionalMessage, @NonNull Class culprit){
        return new BugWithLog(optionalMessage, LogUtil.readApplicationLogs(logLevel), culprit);
    }

    /*
     * Builders
     */
    /**
     * Builds bug information.
     * @param context [REQUIRED] Application context.
     * @param logLevel [REQUIRED] The log level to use while recording the application's logs.
     * @param appName [OPTIONAL] The application name.
     * @param packageName [OPTIONAL] The application's package name.
     * @param versionName [OPTIONAL] The application version name.
     * @param versionCode [OPTIONAL] The application version code.
     */
    public static LogBugBuilder build(@NonNull Context context,
                                      @NonNull String logLevel,
                                      @Nullable String appName,
                                      @Nullable String packageName,
                                      @Nullable String versionName,
                                      int versionCode){
        if(!Utils.isLibraryInitialized(TAG) || Utils.isContextNull(TAG, context))
            return null;

        DeviceInfo dev = gatherDeviceInfo(context);
        if(dev == null){
            Utils.logE(TAG, ErrorMessages.DEV_INFO_COULD_NOT_GATHER);
            return null;
        }
        OSInfo osInfo = gatherOSInfo();
        ApplicationInfo appInfo = gatherAppInfo(appName, packageName, versionName, versionCode);
        BugWithLog bug = gatherBugInfo(logLevel);
        return new LogBugBuilder(dev, osInfo, appInfo, bug);
    }
    /**
     * Builds bug information.
     * @param context [REQUIRED] Application context.
     * @param logLevel [REQUIRED] The log level to use while recording the application's logs.
     * @param appName [OPTIONAL] The application name.
     * @param packageName [OPTIONAL] The application's package name.
     * @param versionName [OPTIONAL] The application version name.
     * @param versionCode [OPTIONAL] The application version code.
     * @param message [OPTIONAL] A message to clarify the bug.
     */
    public static LogBugBuilder build(@NonNull Context context,
                                      @NonNull String logLevel,
                                      @Nullable String appName,
                                      @Nullable String packageName,
                                      @Nullable String versionName,
                                      int versionCode,
                                      @NonNull String message){
        if(!Utils.isLibraryInitialized(TAG) || Utils.isContextNull(TAG, context))
            return null;

        DeviceInfo dev = gatherDeviceInfo(context);
        if(dev == null){
            Utils.logE(TAG, ErrorMessages.DEV_INFO_COULD_NOT_GATHER);
            return null;
        }
        OSInfo osInfo = gatherOSInfo();
        ApplicationInfo appInfo = gatherAppInfo(appName, packageName, versionName, versionCode);
        BugWithLog bug = gatherBugInfo(logLevel, message);
        return new LogBugBuilder(dev, osInfo, appInfo, bug);
    }
    /**
     * Builds bug information.
     * @param context [REQUIRED] Application context.
     * @param logLevel [REQUIRED] The log level to use while recording the application's logs.
     * @param appName [OPTIONAL] The application name.
     * @param packageName [OPTIONAL] The application's package name.
     * @param versionName [OPTIONAL] The application version name.
     * @param versionCode [OPTIONAL] The application version code.
     * @param optionalMessage [OPTIONAL] A message to clarify the bug.
     * @param culprit [OPTIONAL] The {@link Class} responsible for this bug.
     */
    public static LogBugBuilder build(@NonNull Context context,
                                      @NonNull String logLevel,
                                      @Nullable String appName,
                                      @Nullable String packageName,
                                      @Nullable String versionName,
                                      int versionCode,
                                      @Nullable String optionalMessage,
                                      @NonNull Class culprit){
        if(!Utils.isLibraryInitialized(TAG) || Utils.isContextNull(TAG, context))
            return null;

        DeviceInfo dev = gatherDeviceInfo(context);
        if(dev == null){
            Utils.logE(TAG, ErrorMessages.DEV_INFO_COULD_NOT_GATHER);
            return null;
        }
        OSInfo osInfo = gatherOSInfo();
        ApplicationInfo appInfo = gatherAppInfo(appName, packageName, versionName, versionCode);
        BugWithLog bug = gatherBugInfo(logLevel, optionalMessage, culprit);
        return new LogBugBuilder(dev, osInfo, appInfo, bug);
    }

    /*
     * Getters - Setters
     */
    public String getLogLevel() {return logLevel;}
    public synchronized LogBugBuilder setLogLevel(String logLevel) {this.logLevel = logLevel;return this;}
    public String getAppName() {return appName;}
    public synchronized LogBugBuilder setAppName(String appName) {this.appName = appName;return this;}
    public String getPackageName() {return packageName;}
    public synchronized LogBugBuilder setPackageName(String packageName) {this.packageName = packageName;return this;}
    public String getVersionName() {return versionName;}
    public synchronized LogBugBuilder setVersionName(String versionName) {this.versionName = versionName;return this;}
    public int getVersionCode() {return versionCode;}
    public synchronized LogBugBuilder setVersionCode(int versionCode) {this.versionCode = versionCode;return this;}
    public String getOptionalMessage() {return optionalMessage;}
    public synchronized LogBugBuilder setOptionalMessage(String optionalMessage) {this.optionalMessage = optionalMessage;return this;}
    public Class getCulprit() {return culprit;}
    public synchronized LogBugBuilder setCulprit(Class culprit) {this.culprit = culprit;return this;}
}
