package org.aerogear.mobile.core.reactive;

import java.util.concurrent.atomic.AtomicReference;

/**
 * This class implements {@link Request#map(MapFunction)} support for the reactive APIs.
 *
 * @param <T> The type of the request before the map operation
 * @param <R> The type of the request after the map operation.
 */
class MapRequest<T, R> extends AbstractRequest<R> {

    private final AbstractRequest<T> delegateTo;
    private final MapFunction<? super T, ? extends R> mapper;

    public MapRequest(AbstractRequest<T> delegateTo, MapFunction<? super T, ? extends R> mapper) {
        this.mapper = mapper;
        this.delegateTo = delegateTo;
    }

    @Override
    public Request<R> respondWithActual(AtomicReference<Responder<R>> responderRef) {
        delegateTo.respondWithActual(new AtomicReference<>(new Responder<T>() {
            @Override
            public void onResult(T value) {
                Responder<R> responder = responderRef.get();
                if (responder != null) {
                    R mappedValue = null;
                    /*
                     * This may look weird but we are keeping the exception handling contract in
                     * mind. Mapper is *technically* part of the request so exceptions should be
                     * passed to the Responder. However, we don't want to pass exceptions generated
                     * by the responder to itself.
                     *
                     * Therefore we have to return after we catch an exception generated by a
                     * mapper.
                     */
                    try {
                        mappedValue = mapper.map(value);
                    } catch (Exception exception) {
                        onException(exception);
                        return;
                    } finally {
                        // We are done with the original value, it is safe to cleanup.
                        delegateTo.liftCleanupAction().cleanup();
                    }

                    responder.onResult(mappedValue);


                }
            }

            @Override
            public void onException(Exception exception) {
                Responder<R> responder = responderRef.get();
                if (responder != null) {
                    responder.onException(exception);
                }
            }
        }));
        return this;
    }

    @Override
    public void cancel() {
        delegateTo.cancel();
    }

    @Override
    public Request<R> cancelWith(Canceller canceller) {
        delegateTo.cancelWith(canceller);
        return this;
    }

    /**
     * As the Map is responsible for getting a value from the underlying system, it does not need to
     * lift its delegate's cleanup. Future versions may give map the ability to cleanup its own
     * requests.
     *
     * @return an empty cleaner.
     */
    @Override
    protected Cleaner liftCleanupAction() {
        return () -> {
        };
    }
}
