/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import java.util.Objects;
import java.util.UUID;
import terraml.commons.math.Interval;
import terraml.commons.tuple.LatlonEntry;
import terraml.geospatial.Crosstracks;
import terraml.geospatial.Distance;
import terraml.geospatial.DistanceCalculator;
import terraml.geospatial.DistanceNode;
import terraml.geospatial.GeoSegment;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.Latitude;
import terraml.geospatial.Latlon;
import terraml.geospatial.LatlonIntersection;
import terraml.geospatial.Locate;
import terraml.geospatial.Longitude;
import terraml.geospatial.Offset;
import terraml.geospatial.SegmentIntersection;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableGeoSegment implements GeoSegment, Serializable {

    private static final long serialVersionUID = 181265321L;

    public final String id;
    public final Latlon source;
    public final Latlon target;

    public ImmutableGeoSegment(String id, Latlon source, Latlon target) {
        this.id = id;
        this.source = source;
        this.target = target;
    }

    /**
     * @param Latlon
     * @param Latlon
     */
    public ImmutableGeoSegment(Latlon source, Latlon target) {
        this.source = source;
        this.target = target;
        this.id = UUID.randomUUID().toString();
    }

    /**
     * @param Latitude
     * @param Longitude
     * @param Latitude
     * @param Longitude
     */
    public ImmutableGeoSegment(Latitude lat0, Longitude lon0, Latitude lat1, Longitude lon1) {
        this(new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param id
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     */
    public ImmutableGeoSegment(String id, Latitude lat0, Longitude lon0, Latitude lat1, Longitude lon1) {
        this(new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     */
    public ImmutableGeoSegment(double lat0, double lon0, double lat1, double lon1) {
        this(new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param id
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     */
    public ImmutableGeoSegment(String id, double lat0, double lon0, double lat1, double lon1) {
        this(new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param GeoSegment
     */
    public ImmutableGeoSegment(GeoSegment geoSegment) {
        this(geoSegment.getSource(), geoSegment.getTarget());
    }

    //müzik değil acılardır evrensel olan.
    @Override
    public Latlon intersection(GeoSegment segment, DistanceNode tolerance) {

        // find intersection from SegmentIntersection.
        LatlonEntry entry = SegmentIntersection.intersectionOf(this, segment);

        // if there is intersection;
        if (entry != null) {

            // define intersection point
            Latlon crd = new ImmutableLatlon(entry.lat(), entry.lon());

            // avoid bearing intersections.
            if (isOn(crd, tolerance) && segment.isOn(crd, tolerance)) {
                return crd;
            }
        }

        return null;

    }

    @Override
    public boolean intersects(GeoSegment segment) {
        return SegmentIntersection.intersects(this, segment);
    }

    @Override
    public boolean intersects(GeoSegment segment, DistanceNode tolerance) {
        return SegmentIntersection.intersects(this, segment, tolerance);
    }

    @Override
    public GeoSegment offset(DistanceNode distanceNode) {
        Latlon src = Offset.offsetSource(source, target, distanceNode.asMeter());

        return new ImmutableGeoSegment(src, target);
    }

    @Override
    public DistanceNode vinCrosstrack(Latlon latlon) {
        return Crosstracks.crossTrackVinDistance(latlon, source, target);
    }

    @Override
    public DistanceNode havCrosstrack(Latlon latlon) {
        return Crosstracks.crossTrackHavDistance(latlon, source, target);
    }

    @Override
    public boolean isOn(Latlon latlon, DistanceNode tolerance, DistanceCalculator calculator) {
        Latlon entry = LatlonIntersection.closest(latlon, this, calculator);
        double distance = calculator.distanceOf(latlon, entry).asMeter();

        return distance <= tolerance.asMeter();
    }

    @Override
    public boolean isOn(Latlon latlon, DistanceNode tolerance) {
        return LatlonIntersection.withinAccurate(latlon, this, tolerance.asMeter());
    }

    @Override
    public DistanceNode distance(DistanceCalculator calculator) {
        DistanceCalculator calc;

        if (calculator == null) {
            calc = new Distance.Vincenty();
        } else {
            calc = calculator;
        }

        return calc.distanceOf(source, target);
    }

    @Override
    public Latlon closestOf(Latlon latlon, DistanceCalculator calculator) {
        DistanceCalculator calc;

        if (calculator == null) {
            calc = new Distance.Vincenty();
        } else {
            calc = calculator;
        }

        return LatlonIntersection.closest(latlon, this, calc);
    }

    @Override
    public boolean inZone(Latlon latlon) {
        return LatlonIntersection.inSegmentBounds(latlon, this);
    }

    @Override
    public Latlon halfway() {
        return new ImmutableLatlon(Locate.halfWay(this.source, this.target));
    }

    @Override
    public Latlon at(double percentage) {
        return new ImmutableLatlon(Locate.locateWith(this.source, this.target, percentage));
    }

    @Override
    public Latlon getSource() {
        return new ImmutableLatlon(this.source);
    }

    @Override
    public Latlon getTarget() {
        return new ImmutableLatlon(this.target);
    }

    @Override
    public Latlon[] toArray() {
        return new Latlon[]{
            new ImmutableLatlon(this.source),
            new ImmutableLatlon(this.target)
        };
    }

    @Override
    public double[] toDoubleArray() {
        return new double[]{
            source.getLatitude().toDegree(),
            source.getLongitude().toDegree(),
            target.getLatitude().toDegree(),
            target.getLongitude().toDegree()
        };
    }

    @Override
    public ImmutableGeoSegment clone() {
        return new ImmutableGeoSegment(source, target);
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.GeoSegment;
    }

    @Override
    public Latlon[] getBounds() {
        final Interval intValX = new Interval(source.getLatitude().toDegree(), target.getLatitude().toDegree());
        final Interval intValY = new Interval(source.getLongitude().toDegree(), target.getLongitude().toDegree());

        return new Latlon[]{
            new ImmutableLatlon(intValX.left, intValY.left),
            new ImmutableLatlon(intValX.right, intValY.right)
        };
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 23 * hash + Objects.hashCode(this.source);
        hash = 23 * hash + Objects.hashCode(this.target);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableGeoSegment other = (ImmutableGeoSegment) obj;
        if (!Objects.equals(this.source, other.source)) {
            return false;
        }
        if (!Objects.equals(this.target, other.target)) {
            return false;
        }
        return true;
    }
}
