/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import static java.lang.Math.abs;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.Ints;
import static terraml.commons.Objects.isNull;
import terraml.geospatial.GeoBoundingBox;
import terraml.geospatial.GeoPolygon;
import terraml.geospatial.GeoSegment;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.GeoVector;
import terraml.geospatial.Latlon;
import terraml.geospatial.SegmentIntersection;

// sokağa döner.

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableGeoPolygon implements GeoPolygon, Serializable  {

    public final String id;
    public final List<Latlon> vertices;

    public ImmutableGeoPolygon(String id, List<Latlon> vertices) {
        this.id = id;
        this.vertices = vertices;
    }

    /**
     * @param List<Latlon> 
     */
    public ImmutableGeoPolygon(List<Latlon> vertices) {
        if (!isValid(vertices)) {
            throw new IllegalArgumentException("there must be at least 3 vertices in the given list/array.");
        }

        this.vertices = new ArrayList<>(vertices);
        this.id = null;
    }

    /**
     * @param Latlon... 
     */
    public ImmutableGeoPolygon(Latlon... vertices) {
        this(Arrays.asList(vertices));
    }
    
    /**
     * @param id
     * @param vertices 
     */
    public ImmutableGeoPolygon(String id, Latlon... vertices) {
        this(id, Arrays.asList(vertices));
    }

    /**
     * @param GeoPolygon 
     */
    public ImmutableGeoPolygon(GeoPolygon geoPolygon) {
        this(geoPolygon.toList());
    }

    @Override
    public boolean contains(Latlon coordinate) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polygon is empty.");
        }

        GeoVector _v0 = GeoVector.fromLatlon(coordinate);
        int _len = getVerticesCount() - 1;

        List<GeoVector> _transformed = new ArrayList<>();

        for ( int i = 0; i < _len; i++ ) {
            _transformed.add(_v0.translate(GeoVector.fromLatlon(this.vertices.get(i)).reverse()));
        }

        _transformed.add(_transformed.get(0));

        double _letQ = 0.0d;
        for ( int i = 0; i < _len; i++ ) {
            _letQ += _transformed.get(i).angleTo(_transformed.get(i + 1), _v0).radian;
        }

        return isGreater(abs(_letQ), Math.PI);
    }

    @Override
    public boolean contains(GeoPolygon polygon) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Either this or given polygon is empty.");
        }
        
        if (isNull(polygon)) {
            throw new IllegalStateException("Either this or given polygon is empty.");
        }
        
        List<Latlon> argVertx = polygon.getVertices();
        
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(this.vertices);
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(argVertx);

        // check given is inside bounds of this
        // and
        // not intersects it's segments with this
        if (!_selfRect.contains(_polyRect) || intersects(polygon)) {
            return false;
        }

        // check every vertex of given polygon contained in this or not.
        final Collection<Latlon> _ptr = argVertx;
        for ( Latlon curr : _ptr ) {

            // if any vertex is not contained in this, then return false.
            if (!contains(curr)) {
                return false;
            }

        }

        return true;
    }
    
    // kavga benzemez ps'deki combolara.

    @Override
    public boolean intersects(GeoPolygon polygon) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Either this or given polygon is empty.");
        }
        
        if (isNull(polygon)) {
            throw new IllegalStateException("Either this or given polygon is empty.");
        }

        final Collection<GeoSegment> _self = toSegments();
        final Collection<GeoSegment> _ply = polygon.toSegments();

        return _self.stream().anyMatch((curr) -> (_ply.stream().anyMatch((next) -> (SegmentIntersection.intersects(curr, next)))));
    }

    @Override
    public Collection<GeoSegment> toSegments() {
        final Collection<GeoSegment> _sgm = new ArrayList<>();
        final List<Latlon> _vx = this.vertices;

        boolean _clsd = isClosed();
        int _lh = _vx.size();

        for ( int i = 0; i < _lh; i++ ) {

            final int _idx = i + 1;

            if (Ints.isSmaller(_idx, _lh)) {

                final Latlon _src = _vx.get(i);
                final Latlon _tar = _vx.get(_idx);
                final GeoSegment _seg = new ImmutableGeoSegment(_src, _tar);

                _sgm.add(_seg);

            }

        }

        // return segments as closed-polygon
        if (!_clsd) {

            final Latlon _src = _vx.get(_lh - 1);
            final Latlon _tar = _vx.get(0);
            final GeoSegment _seg = new ImmutableGeoSegment(_src, _tar);
            _sgm.add(_seg);

        }

        return _sgm;
    }

    @Override
    public boolean isValid() {
        return Ints.isGreaterEqual(vertices.size(), 3);
    }

    @Override
    public boolean isClosed() {
        final List<Latlon> _lst = this.vertices;
        final Latlon _org = _lst.get(0);
        final int _lh = _lst.size() - 1;

        return _lst.get(_lh).equals(_org);
    }

    @Override
    public boolean isProper() {
        return isValid() && isClosed();
    }

    @Override
    public boolean prepare() {
        if (Ints.isEqual(vertices.size(), 3)) {
            
            this.vertices.add(this.vertices.get(0));
            
            return this.vertices.get(0).equals(this.vertices.get(getVerticesCount() - 1));
        }
        
        if (isValid() && !isClosed()) {
            
            this.vertices.add(this.vertices.get(0));

            return this.vertices.get(0).equals(this.vertices.get(getVerticesCount() - 1));
            
        }

        return false;
    }

    /**
     * @param boolean
     * @param boolean
     * @return 
     */
    private int _lowUpp(boolean isMax, boolean isX) {
        double ref = isMax ? 0 : Double.MAX_VALUE;
        int idx = -1;

        for (int t = 0; t < getVerticesCount(); t++) {
            final double temp = isX ? vertices.get(t).getLatitude().toDegree() : vertices.get(t).getLongitude().toDegree();
            if (isMax) {
                if (temp > ref) {
                    idx = t;
                    ref = temp;
                }
            } else {
                if (temp < ref) {
                    idx = t;
                    ref = temp;
                }
            }
        }

        return idx;
    }

    @Override
    public GeoPolygon addVertex(Latlon vertex) {
        final List<Latlon> _vertx = new ArrayList<>(vertices);

        _vertx.add(vertex);

        // make it closed form
        final Latlon originLatlon = _vertx.get(0);
        _vertx.add(originLatlon);

        return new ImmutableGeoPolygon(_vertx);
    }

    @Override
    public GeoPolygon removeVertex(Latlon vertex) {
        // don't let it if there is 4 vertices left. Consider throwing an exception here.
        if (isSmaller(getVerticesCount(), 4)) {
            return this;
        }

        final List<Latlon> _vertx = new ArrayList<>(vertices);

        // if requested vertex is origin of polygon, then re-scale closed form.
        if (vertex.equals(vertices.get(0))) {
            final Latlon newOri = vertices.get(1);
            final int len = getVerticesCount();
            _vertx.remove(0);
            _vertx.remove(len - 2);
            _vertx.add(newOri);
        }

        _vertx.remove(vertex);

        return new ImmutableGeoPolygon(_vertx);
    }

    @Override
    public List<Latlon> getVertices() {
        return new ArrayList<>(this.vertices);
    }

    @Override
    public Latlon[] toArray() {
        final int len = getVerticesCount();
        final Latlon[] asArray = new Latlon[len];

        for (int i = 0; i < len; i++) {
            asArray[i] = new ImmutableLatlon(vertices.get(i));
        }

        return asArray;
    }

    @Override
    public List<Latlon> toList() {
        return new ArrayList<>(vertices);
    }

    @Override
    public int getVerticesCount() {
        return vertices.size();
    }

    @Override
    public ImmutableGeoPolygon clone() {
        return new ImmutableGeoPolygon(vertices);
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.GeoPolygon;
    }

    @Override
    public Latlon[] getBounds() {
        final double uppLat = vertices.get(_lowUpp(true, true)).getLatitude().toDegree();
        final double uppLon = vertices.get(_lowUpp(false, true)).getLongitude().toDegree();
        final double lowLat = vertices.get(_lowUpp(true, false)).getLatitude().toDegree();
        final double lowLon = vertices.get(_lowUpp(false, false)).getLongitude().toDegree();

        final Latlon upp = new ImmutableLatlon(lowLat, lowLon);
        final Latlon low = new ImmutableLatlon(uppLat, uppLon);

        return new Latlon[]{upp, low};
    }

    @Override
    public String getId() {
        return id;
    }
    
    /**
     * @param _lst
     * @return 
     */
    private static boolean isValid(List<Latlon> _lst) {
        return Ints.isGreaterEqual(_lst.size(), 3);
    }
    
    /**
     * @return true if null or empty. Otherwise false.
     */
    private boolean nullOrEmpty() {
        return this.vertices == null || this.vertices.isEmpty();
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 53 * hash + Objects.hashCode(this.vertices);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableGeoPolygon other = (ImmutableGeoPolygon) obj;
        if (!Objects.equals(this.vertices, other.vertices)) {
            return false;
        }
        return true;
    }
}
