/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.math.Angle;
import terraml.geospatial.Distance;
import terraml.geospatial.DistanceNode;
import terraml.geospatial.GeoCircle;
import terraml.geospatial.GeoPolyline;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.Latitude;
import terraml.geospatial.Latlon;
import terraml.geospatial.LatlonIntersection;
import terraml.geospatial.Longitude;
import terraml.geospatial.Zone;

// tombala, poker, çizik nasıl meşgale ?

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableGeoCircle implements GeoCircle, Serializable  {

    public final String id;
    public final Latlon center;
    public final DistanceNode radius;

    public ImmutableGeoCircle(String id, Latlon center, DistanceNode radius) {
        this.id = id;
        this.center = center;
        this.radius = radius;
    }

    /**
     * @param Latlon
     * @param double 
     */
    public ImmutableGeoCircle(Latlon center, DistanceNode radius) {
        this.center = center;
        this.radius = radius;
        this.id = null;
    }

    /**
     * @param Latitude
     * @param Longitude
     * @param double 
     */
    public ImmutableGeoCircle(Latitude lat0, Longitude lon0, DistanceNode radius) {
        this(new ImmutableLatlon(lat0, lon0), radius);
    }
    
    /**
     * @param id
     * @param lat0
     * @param lon0
     * @param radius 
     */
    public ImmutableGeoCircle(String id, Latitude lat0, Longitude lon0, DistanceNode radius) {
        this(id, new ImmutableLatlon(lat0, lon0), radius);
    }

    /**
     * @param double
     * @param double
     * @param double 
     */
    public ImmutableGeoCircle(double lat0, double lon0, DistanceNode radius) {
        this(new ImmutableLatlon(lat0, lon0), radius);
    }
    
    /**
     * @param id
     * @param lat0
     * @param lon0
     * @param radius 
     */
    public ImmutableGeoCircle(String id, double lat0, double lon0, DistanceNode radius) {
        this(id, new ImmutableLatlon(lat0, lon0), radius);
    }

    /**
     * @param GeoCircle 
     */
    public ImmutableGeoCircle(GeoCircle circle) {
        this(circle.getCenter(), circle.getRadius());
    }

    @Override
    public boolean contains(Latlon coordinate) {
        return LatlonIntersection.withinAccurate(coordinate, this);
    }

    @Override
    public boolean contains(GeoCircle geoCircle) {
        final double _d = Distance.vincenty(this.center, geoCircle.getCenter());
        final double _dr = this.radius.asMeter() - geoCircle.getRadius().asMeter();

        return isSmallerEqual(_d, _dr);
    }

    @Override
    public boolean intersects(GeoCircle circle) {
        final double _d = Distance.vincenty(this.center, circle.getCenter());
        final double _dr = this.radius.asMeter() + circle.getRadius().asMeter();

        return isSmallerEqual(_d, _dr);
    }

    @Override
    public Latlon getCenter() {
        return new ImmutableLatlon(center);
    }

    @Override
    public DistanceNode getRadius() {
        return this.radius;
    }
    
    @Override
    public GeoCircle offset(DistanceNode distanceNode, Angle bearing) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public double area() {
        return Zone.areaOf(this);
    }

    @Override
    public GeoPolyline toOctagon() {
        Latlon _center = this.center;

        List<Latlon> _lst = new ArrayList<>();

        _lst.add(_center.offset(this.radius, Angle.fromDegree(0.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(45.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(90.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(135.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(180.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(225.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(270.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(325.0d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(360.0d)));

        return new ImmutableGeoPolyline(_lst);
    }

    @Override
    public GeoPolyline to2xOctagon() {
        Latlon _center = this.center;

        List<Latlon> _lst = new ArrayList<>();

        _lst.add(_center.offset(this.radius, Angle.fromDegree(0.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(22.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(45.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(67.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(90.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(112.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(135.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(157.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(180.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(202.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(225.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(247.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(270.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(297.5d)));
        _lst.add(_center.offset(this.radius, Angle.fromDegree(325.0d)));

        _lst.add(_center.offset(this.radius, Angle.fromDegree(360.0d)));

        return new ImmutableGeoPolyline(_lst);
    }
    
    @Override
    public Latlon[] getBounds() {
        final Latlon _ctr = new ImmutableLatlon(this.center);

        // find north
        Latlon _ne = _ctr.offset(this.radius, Angle.fromDegree(0.0d));

        // find south
        Latlon _sw = _ctr.offset(this.radius, Angle.fromDegree(180.0d));

        // offset north to north-east
        _ne = _ne.offset(this.radius, Angle.fromDegree(90.0d));

        // offset south to south-west
        _sw = _sw.offset(this.radius, Angle.fromDegree(270.0d));

        return new Latlon[]{_ne, _sw};
    }

    @Override
    public double[] toArray() {
        return new double[]{center.getLatitude().toDegree(), center.getLongitude().toDegree(), radius.asMeter()};
    }

    @Override
    public ImmutableGeoCircle clone() {
        return new ImmutableGeoCircle(center, radius);
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.GeoCircle;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 97 * hash + Objects.hashCode(this.center);
        hash = 97 * hash + Objects.hashCode(this.radius);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableGeoCircle other = (ImmutableGeoCircle) obj;
        if (!Objects.equals(this.center, other.center)) {
            return false;
        }
        if (!Objects.equals(this.radius, other.radius)) {
            return false;
        }
        return true;
    }
}
