/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Doubles.isSmallerEqual;
import static terraml.commons.Objects.isNull;
import terraml.commons.annotation.Development;
import terraml.commons.annotation.File;
import terraml.commons.math.Angle;
import static terraml.geospatial.Azimuths.northBasedAzimuth;
import static terraml.geospatial.GeoUtils.lat2deg;
import static terraml.geospatial.GeoUtils.lon2deg;
import static terraml.geospatial.LatlonIntersection.closestAccurate;
import terraml.geospatial.impl.ImmutableLatlon;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "SegmentIntersection.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
public final class SegmentIntersection {

    private SegmentIntersection() {
    }

    /**
     * ! Warning : Sometimes method may result as line, not segment. Until this
     * bug fixed use with this LatlonIntersection.withnin.
     *
     * @param geoSegment0
     * @param geoSegment1
     * @return
     */
    @Development(status = Development.Status.EXPERIMENTAL)
    public static Latlon intersectionOf(GeoSegment geoSegment0, GeoSegment geoSegment1) {
        GeoVector _v0 = GeoVector.fromLatlon(geoSegment0.getSource());
        GeoVector _v1 = GeoVector.fromLatlon(geoSegment0.getTarget());
        GeoVector _v2 = GeoVector.fromLatlon(geoSegment1.getSource());
        GeoVector _v3 = GeoVector.fromLatlon(geoSegment1.getTarget());

        GeoVector _crs0 = _v0.cross(_v1);
        GeoVector _crs1 = _v2.cross(_v3);

        GeoVector i1 = _crs0.cross(_crs1);
        GeoVector i2 = _crs1.cross(_crs0);

        GeoVector _ttl = _v0.translate(_v1).translate(_v2).translate(_v3);
        GeoVector _intrsct = isGreater(_ttl.dot(i1), 0) ? i1 : i2;

        Latlon temp = null;
        if (isNull(_intrsct)) {
            temp = null;
        } else {
            try {
                temp = _intrsct.toLatlon();
            } catch (Exception e) {
                temp = null;
            }
        }

        return temp;
    }

    /**
     * Uses triangulation. Seems working.
     *
     * @param geoSegment0
     * @param geoSegment1
     * @param toleranceM
     * @return
     */
    @Development(status = Development.Status.STABLE)
    public static boolean intersects(GeoSegment geoSegment0, GeoSegment geoSegment1, DistanceNode tolerance) {
        final Angle _bearing0 = northBasedAzimuth(geoSegment0.getSource(), geoSegment0.getTarget());
        final Angle _bearing1 = northBasedAzimuth(geoSegment1.getSource(), geoSegment1.getTarget());

        final Latlon _tried = Triangulation.triangulate(geoSegment0.getSource(), _bearing0, geoSegment1.getSource(), _bearing1);
        if (isNull(_tried)) {
            return false;
        }

        final double _crssTrck0 = Distance.vincenty(_tried, closestAccurate(_tried, geoSegment0));
        final double _crssTrck1 = Distance.vincenty(_tried, closestAccurate(_tried, geoSegment1));

        return isSmaller(Math.abs(_crssTrck0 - _crssTrck1), tolerance.asMeter());
    }

    /**
     * gördüler, yaşam nedir gördüler,
     * bir umuda yürüyenler ulu orta öldüler.
     * en güzeli sevdiler,
     * gökyüzüne bakmayı ve hep yarına güldüler.
     */
    /**
     * Default 1 meter tolerance.
     *
     * @param geoSegment0
     * @param geoSegment1
     * @return
     */
    public static boolean intersects(GeoSegment geoSegment0, GeoSegment geoSegment1) {
        return intersects(geoSegment0, geoSegment1, DistanceNode.fromMeter(1.0d));
    }

    /**
     * Don't use for now.
     *
     * @param geoSegment0
     * @param geoSegment1
     * @return
     */
    @Development(status = Development.Status.EXPERIMENTAL)
    public static Latlon intersectionOf2(GeoSegment geoSegment0, GeoSegment geoSegment1) {
        final double Qx = lat2deg(geoSegment0.getTarget()) - lat2deg(geoSegment0.getSource());
        final double Qy = lon2deg(geoSegment0.getSource()) - lon2deg(geoSegment0.getTarget());
        final double Qt = Qx * lon2deg(geoSegment0.getSource()) + Qy * lat2deg(geoSegment0.getSource());

        final double Wx = lat2deg(geoSegment1.getTarget()) - lat2deg(geoSegment1.getSource());
        final double Wy = lon2deg(geoSegment1.getSource()) - lon2deg(geoSegment1.getTarget());
        final double Wt = Wx * lon2deg(geoSegment1.getSource()) + Wy * lat2deg(geoSegment1.getSource());

        double determinate = Qx * Wy - Wx * Qy;
        Latlon intersection = null;

        if (determinate != 0) {
            double Zx = (Wy * Qt - Qy * Wt) / determinate;
            double Zy = (Qx * Wt - Wx * Qt) / determinate;

            Latlon intersect = new ImmutableLatlon(Zy, Zx);
            final boolean Vx = _inBBox(geoSegment0.getSource(), geoSegment0.getTarget(), intersect);
            final boolean Vy = _inBBox(geoSegment1.getSource(), geoSegment1.getTarget(), intersect);

            if (Vx && Vy) {
                intersection = intersect;
            } else {
                intersection = null;
            }
        }

        return intersection;
    }

    /**
     * @param Latlon
     * @param Latlon
     * @param Latlon
     * @return
     */
    private static boolean _inBBox(Latlon latlon0, Latlon latlon1, Latlon latlon2) {
        boolean _latB;
        boolean _lonB;

        if (isSmaller(lat2deg(latlon0), lat2deg(latlon1))) {
            _latB = isSmallerEqual(lat2deg(latlon0), lat2deg(latlon2)) && isGreaterEqual(lat2deg(latlon1), lat2deg(latlon2));
        } else {
            _latB = isGreaterEqual(lat2deg(latlon0), lat2deg(latlon2)) && isSmallerEqual(lat2deg(latlon1), lat2deg(latlon2));
        }

        if (isSmaller(lon2deg(latlon0), lon2deg(latlon1))) {
            _lonB = isSmallerEqual(lon2deg(latlon0), lon2deg(latlon2)) && isGreaterEqual(lon2deg(latlon1), lon2deg(latlon2));
        } else {
            _lonB = isGreaterEqual(lon2deg(latlon0), lon2deg(latlon2)) && isSmallerEqual(lon2deg(latlon1), lon2deg(latlon2));
        }

        return _latB && _lonB;
    }

    /**
     * algebraic manipulation. use it for small areas. {Seems working}
     *
     * @param geoSegment0
     * @param geoSegment1
     * @return
     */
    public static boolean intersects2(GeoSegment geoSegment0, GeoSegment geoSegment1) {
        final double p0_x = lat2deg(geoSegment0.getSource());
        final double p0_y = lon2deg(geoSegment0.getSource());
        final double p1_x = lat2deg(geoSegment0.getTarget());
        final double p1_y = lon2deg(geoSegment0.getTarget());

        final double p2_x = lat2deg(geoSegment1.getSource());
        final double p2_y = lon2deg(geoSegment1.getSource());
        final double p3_x = lat2deg(geoSegment1.getTarget());
        final double p3_y = lon2deg(geoSegment1.getTarget());

        final double s1_x = p1_x - p0_x;
        final double s1_y = p1_y - p0_y;
        final double s2_x = p3_x - p2_x;
        final double s2_y = p3_y - p2_y;

        double s = (-s1_y * (p0_x - p2_x) + s1_x * (p0_y - p2_y)) / (-s2_x * s1_y + s1_x * s2_y);
        double t = (s2_x * (p0_y - p2_y) - s2_y * (p0_x - p2_x)) / (-s2_x * s1_y + s1_x * s2_y);

        return s >= 0 && s <= 1 && t >= 0 && t <= 1;
    }
}
