/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.asin;
import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import terraml.commons.annotation.Development;
import terraml.commons.math.Angle;
import static terraml.geospatial.Azimuths.northBasedAzimuth;
import static terraml.geospatial.GeoUtils.fixLongitudeFromDegree;
import static terraml.geospatial.GeoUtils.lat2rad;
import static terraml.geospatial.GeoUtils.lon2rad;
import terraml.geospatial.impl.ImmutableLatlon;

// Mavzerim cahile ateş saçar
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Offset {

    private Offset() {
    }

    /**
     * 14 km için 16 metre hata payı.
     *
     * @param lat       latitude in radian
     * @param lon       longitude in radian
     * @param distanceM distance in meter
     * @param bearing   north based azimuth angle in radian
     * @return calculated lat, lon in degrees.
     */
    public static double[] destinationOfFromRadian(double lat, double lon, double distanceM, double bearing) {
        final double Qt = distanceM / GeoUtils.EARTH_RADIUS_M;

        final double QxSin = sin(lat);
        final double QxCos = cos(lat);
        final double QtSin = sin(Qt);
        final double QtCos = cos(Qt);

        final double QrSin = sin(bearing);
        final double QrCos = cos(bearing);

        final double Wt = (QxSin * QtCos) + (QxCos * QtSin * QrCos);
        double _latQ = asin(Wt);

        final double Wy = QrSin * QtSin * QxCos;
        final double Wx = QtCos - (QxSin * sin(_latQ));
        double _lonQ = lon + atan2(Wy, Wx);

        _latQ = Math.toDegrees(_latQ);
        _lonQ = fixLongitudeFromDegree(Math.toDegrees(_lonQ));

        return new double[]{_latQ, _lonQ};
    }

    /**
     *
     * @param source
     * @param distanceM in meter
     * @param bearing
     * @return
     */
    @Development(status = Development.Status.STABLE)
    public static Latlon destinationOf(Latlon source, double distanceM, Angle bearing) {
        final double[] latlon = destinationOfFromRadian(lat2rad(source), lon2rad(source), distanceM, bearing.radian);

        return new ImmutableLatlon(latlon[0], latlon[1]);
    }

    /**
     *
     * Note: If given distance is bigger than the distance between source and
     * target it returns null. If you would like to find a point that is on the
     * same bearing, use the Locate.locateWith with more than 100 percentage.
     *
     * @param source    source point to be relocated
     * @param target    target point
     * @param distanceM distance in meters
     * @return relocated source
     */
    public static Latlon offsetSource(Latlon source, Latlon target, double distanceM) {
        return Offset.destinationOf(source, distanceM, northBasedAzimuth(source, target));
    }
}
