/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.abs;
import java.util.ArrayList;
import java.util.List;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Doubles.isSmallerEqual;
import static terraml.commons.Objects.isNull;
import static terraml.geospatial.GeoUtils.lat2deg;
import static terraml.geospatial.GeoUtils.lon2deg;

// Uçuyoruz kanatsız rahatta.

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class LatlonIntersection {

    private LatlonIntersection() {
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @return
     */
    public static boolean inSegmentBounds(Latlon latlon, GeoSegment geoSegment) {
        final GeoVector _v0 = GeoVector.fromLatlon(latlon);
        final GeoVector _v1 = GeoVector.fromLatlon(geoSegment.getSource());
        final GeoVector _v2 = GeoVector.fromLatlon(geoSegment.getTarget());

        final GeoVector _v3 = _v0.translate(_v1.reverse());
        final GeoVector _v4 = _v2.translate(_v1.reverse());
        final GeoVector _v5 = _v0.translate(_v2.reverse());
        final GeoVector _v6 = _v1.translate(_v2.reverse());

        final double _let0 = _v3.dot(_v4);
        final double _let1 = _v5.dot(_v6);

        return isGreaterEqual(_let0, 0d) && isGreaterEqual(_let1, 0d);
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @param calculator
     * @return
     */
    public static Latlon closest(Latlon latlon, GeoSegment geoSegment, DistanceCalculator calculator) {
        final DistanceCalculator calc = isNull(calculator) ? new Distance.Vincenty() : calculator;
        
        if (!inSegmentBounds(latlon, geoSegment)) {
            final double _dist0 = calc.distanceOf(latlon, geoSegment.getSource()).distance;
            final double _dist1 = calc.distanceOf(latlon, geoSegment.getTarget()).distance;
            if (isGreater(_dist0, _dist1)) {
                return geoSegment.getTarget();
            } else {
                return geoSegment.getSource();
            }
        }

        final GeoVector _v0 = GeoVector.fromLatlon(latlon);
        final GeoVector _v1 = GeoVector.fromLatlon(geoSegment.getSource());
        final GeoVector _v2 = GeoVector.fromLatlon(geoSegment.getTarget());

        final GeoVector _V3 = _v1.cross(_v2);
        final GeoVector _V4 = _v0.cross(_V3);

        return _V3.cross(_V4).toLatlon();
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @return
     */
    public static Latlon closestFast(Latlon latlon, GeoSegment geoSegment) {
        return closest(latlon, geoSegment, new Distance.Haversine());
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @return
     */
    public static Latlon closestAccurate(Latlon latlon, GeoSegment geoSegment) {
        return closest(latlon, geoSegment, new Distance.Vincenty());
    }

    /**
     *
     * @param latlon
     * @param geoPolygon
     * @return
     */
    public static boolean withinFast(Latlon latlon, GeoPolygon geoPolygon) {
        boolean _cross = false;

        List<Latlon> _vertex = geoPolygon.toList();
        int _len = _vertex.size();
        _vertex.remove(_len - 1);

        for (int i = 0; i < _len - 1; i++) {
            int _opp = _len - 1;

            double _plon0 = lon2deg(_vertex.get(i));
            double _plon1 = lon2deg(_vertex.get(_opp));
            double _lon0 = lon2deg(latlon);

            boolean _let0 = isSmallerEqual(_plon0, _lon0) && isSmaller(_lon0, _plon1);
            boolean _let1 = isSmallerEqual(_plon1, _lon0) && isSmaller(_lon0, _plon0);

            if (_let0 || _let1) {

                double _plat0 = lat2deg(_vertex.get(i));
                double _plat1 = lat2deg(_vertex.get(_opp));
                double _lat0 = lat2deg(latlon);

                double _letQ = _plat1 - _plat0;

                _letQ *= _lon0 - _plon0;
                _letQ /= _plon1 - _plon0;
                _letQ += _plat0;

                if (isSmaller(_lat0, _letQ)) {
                    _cross = !_cross;
                }

            }

        }

        return _cross;
    }

    /**
     *
     * @param latlon
     * @param geoPolygon
     * @return
     */
    public static boolean withinAccurate(Latlon latlon, GeoPolygon geoPolygon) {
        GeoVector _v0 = GeoVector.fromLatlon(latlon);

        int _len = geoPolygon.toList().size() - 1;

        List<GeoVector> _transformed = new ArrayList<>();

        for ( int i = 0; i < _len; i++ ) {
            _transformed.add(_v0.translate(GeoVector.fromLatlon(geoPolygon.toList().get(i)).reverse()));
        }

        _transformed.add(_transformed.get(0));

        double _letQ = 0.0d;
        for ( int i = 0; i < _len; i++ ) {
            _letQ += _transformed.get(i).angleTo(_transformed.get(i + 1), _v0).radian;
        }

        return isGreater(abs(_letQ), Math.PI);
    }

    /**
     *
     * @param latlon
     * @param geoCircle
     * @param calculator
     * @return
     */
    public static boolean within(Latlon latlon, GeoCircle geoCircle, DistanceCalculator calculator) {
        return isSmaller(calculator.distanceOf(latlon, geoCircle.getCenter()).distance, geoCircle.getRadius().asMeter());
    }

    /**
     *
     * @param latlon
     * @param geoCircle
     * @return
     */
    public static boolean withinFast(Latlon latlon, GeoCircle geoCircle) {
        return within(latlon, geoCircle, new Distance.Haversine());
    }

    /**
     *
     * @param latlon
     * @param geoCircle
     * @return
     */
    public static boolean withinAccurate(Latlon latlon, GeoCircle geoCircle) {
        return within(latlon, geoCircle, new Distance.Vincenty());
    }

    /**
     *
     * @param latlon
     * @param geoBoundingBox
     * @return
     */
    public static boolean within(Latlon latlon, GeoBoundingBox geoBoundingBox) {
        return isGreaterEqual(lat2deg(latlon), lat2deg(geoBoundingBox.getSouthWest()))
                && isGreaterEqual(lon2deg(latlon), lon2deg(geoBoundingBox.getSouthWest()))
                && isSmallerEqual(lat2deg(latlon), lat2deg(geoBoundingBox.getNorthEast()))
                && isSmallerEqual(lon2deg(latlon), lon2deg(geoBoundingBox.getNorthEast()));
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @param tolaranceMet
     * @return
     */
    public static boolean withinFast(Latlon latlon, GeoSegment geoSegment, double tolaranceMet) {
        return isSmallerEqual(Distance.haversine(latlon, closestFast(latlon, geoSegment)), tolaranceMet);
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @param tolaranceMet
     * @return
     */
    public static boolean withinAccurate(Latlon latlon, GeoSegment geoSegment, double tolaranceMet) {
        return isSmallerEqual(Distance.vincenty(latlon, closestFast(latlon, geoSegment)), tolaranceMet);
    }

    /**
     *
     * @param latlon
     * @param geoSegment
     * @param calculator
     * @param tolaranceMet
     * @return
     */
    public static boolean within(Latlon latlon, GeoSegment geoSegment, DistanceCalculator calculator, double tolaranceMet) {
        return isSmallerEqual(calculator.distanceOf(latlon, closestFast(latlon, geoSegment)).distance, tolaranceMet);
    }
}
