/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package terraml.geospatial;

import java.util.Collection;
import java.util.LinkedList;
import static terraml.commons.Objects.isNull;
import static terraml.commons.Objects.nonNull;
import terraml.geospatial.impl.ImmutableGeoBoundingBox;

// burda çocuklar hep tırnak yiyerek doyar.

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 * 
 */
public final class Intersect {

    private Intersect() {
    }

    /**
     * @param circle
     * @param rectangle
     * @return true if any intersection exists. Otherwise false.
     */
    public static boolean anyFair(GeoCircle circle, GeoBoundingBox rectangle) {
        final GeoBoundingBox _crclRect = new ImmutableGeoBoundingBox(circle.getBounds());

        // avoid further calculations
        if (disjoint(_crclRect, rectangle)) {
            return false;
        }

        // export segments of both
        Collection<GeoSegment> _crclSeg = _crclRect.toHeuristicSegments();
        Collection<GeoSegment> _rectSeg = rectangle.toHeuristicSegments();

        // determine any exising intersection between segments.
        if (!any(_crclSeg, _rectSeg)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param rect0
     * @param rect1
     * @param toleranceMeter
     * @return intersection points between given GeoBoundingBox and GeoBoundingBox. If there is no intersection,
     * then empty LinkedList.
     */
    public static Collection<Latlon> collect(GeoBoundingBox rect0, GeoBoundingBox rect1, DistanceNode tolerance) {
        final Collection<Latlon> _ints = new LinkedList<>();

        // avoid further calculations
        if (!rect0.intersects(rect1)) {
            return _ints;
        }

        // export segments of both
        final Collection<GeoSegment> _rect0Seg = rect0.toHeuristicSegments();
        final Collection<GeoSegment> _rect1Seg = rect1.toHeuristicSegments();

        // intersection if any intersection exists
        final Collection<Latlon> _collected = collect(_rect0Seg, _rect1Seg, tolerance);

        //
        if (_collected.isEmpty()) {
            return _ints;
        } else {
            _ints.addAll(_collected);
        }

        return _ints;
    }
    
    /**
     * @param poly0
     * @param poly1
     * @param tolerance
     * @return intersection points between given GeoPolygon and GeoPolygon. If there is no intersection,
     * then empty LinkedList.
     */
    public static Collection<Latlon> collect(GeoPolygon poly0, GeoPolygon poly1, DistanceNode tolerance) {
        final Collection<Latlon> _ints = new LinkedList<>();

        // geopolygon segments
        final Collection<GeoSegment> _poly0Seg = poly0.toSegments();
        
        // geopolygon segments
        final Collection<GeoSegment> _poly1Seg = poly1.toSegments();

        // intersection points
        final Collection<Latlon> _collected = collect(_poly0Seg, _poly1Seg, tolerance);

        if (_collected.isEmpty()) {
            return _ints;
        } else {
            _ints.addAll(_collected);
        }

        return _ints;
    }
    
    /**
     * @param poly0
     * @param rect0
     * @param tolerance
     * @return intersection points between given GeoPolygon and GeoBoundingBox. If there is no intersection,
     * then empty LinkedList.
     */
    public static Collection<Latlon> collect(GeoPolygon poly0, GeoBoundingBox rect0, DistanceNode tolerance) {
        final Collection<Latlon> _ints = new LinkedList<>();

        // geopolygon segments
        final Collection<GeoSegment> _poly0Seg = poly0.toSegments();
        
        // geobox segments
        final Collection<GeoSegment> _rect0Seg = rect0.toHeuristicSegments();

        // intersection points
        final Collection<Latlon> _collected = collect(_poly0Seg, _rect0Seg, tolerance);

        if (_collected.isEmpty()) {
            return _ints;
        } else {
            _ints.addAll(_collected);
        }

        return _ints;
    }
    
    /**
     * @param poly0
     * @param poly1
     * @param tolerance
     * @return intersection points between given GeoPolygon and GeoPolyline. If there is no intersection,
     * then empty LinkedList.
     */
    public static Collection<Latlon> collect(GeoPolygon poly0, GeoPolyline poly1, DistanceNode tolerance) {
        final Collection<Latlon> _ints = new LinkedList<>();

        // geopolygon segments
        final Collection<GeoSegment> _poly0Seg = poly0.toSegments();
        
        // geopolyline segments
        final Collection<GeoSegment> _rect0Seg = poly1.toSegments();

        // intersection points
        final Collection<Latlon> _collected = collect(_poly0Seg, _rect0Seg, tolerance);

        if (_collected.isEmpty()) {
            return _ints;
        } else {
            _ints.addAll(_collected);
        }

        return _ints;
    }
    
    /**
     * @param poly0
     * @param rect0
     * @param toleranceMeter
     * @return intersection points between given GeoPolyline and GeoBoundingBox. If there is no intersection,
     * then empty LinkedList.
     */
    public static Collection<Latlon> collect(GeoPolyline poly0, GeoBoundingBox rect0, DistanceNode tolerance) {
        final Collection<Latlon> _ints = new LinkedList<>();

        // polyline segments
        final Collection<GeoSegment> _poly0Seg = poly0.toSegments();
        
        // geobox segments
        final Collection<GeoSegment> _rect0Seg = rect0.toHeuristicSegments();

        // intersections
        final Collection<Latlon> _collected = collect(_poly0Seg, _rect0Seg, tolerance);

        if (_collected.isEmpty()) {
            return _ints;
        } else {
            _ints.addAll(_collected);
        }

        return _ints;
    }
    
    /**
     * @param rect0
     * @param rect1
     * @return true if given geoboxes are disjoint. Otherwise false.
     */
    public static boolean disjoint(GeoBoundingBox rect0, GeoBoundingBox rect1) {
        // determine does any contains other one
        if (rect0.contains(rect1) || rect1.contains(rect0)) {
            return false;
        }

        // do they intersects
        return !rect0.intersects(rect1);
    }
    
    /**
     * @param c0
     * @param c1
     * @param tolerance
     * @return true if given c1 is on the c0. Otherwise false.
     */
    public static boolean isOn(Latlon c0, Latlon c1, DistanceNode tolerance) {
        return new Distance.Vincenty().distanceOf(c0, c1).asMeter() <= tolerance.asMeter();
    }
    
    /**
     * @param object
     * @param coordinate
     * @param toleranceMeter
     * @return true if given Latlon is on the given GeoCircle. Otherwise false.
     */
    public static boolean isOn(GeoCircle object, Latlon coordinate, DistanceNode tolerance) {
        double distance = new Distance.Vincenty().distanceOf(object.getCenter(), coordinate).asMeter();
        double radiusMeter = object.getRadius().asMeter();

        double lo = radiusMeter - tolerance.asMeter();
        double hi = radiusMeter + tolerance.asMeter();

        return distance <= hi && distance >= lo;
    }
    
    /**
     * @param seg0
     * @param coordinate
     * @param toleranceMeter
     * @return true if given Latlon is on any of the given collection of segments. Otherwise false.
     */
    public static boolean anyOn(Collection<GeoSegment> seg0, Latlon coordinate, DistanceNode tolerance) {
        if (isNull(seg0)) {
            return false;
        }

        // for each segment
        for ( GeoSegment segment : seg0 ) {
            
            // is coordinate on segment.
            if (segment.isOn(coordinate, tolerance)) {
                return true;
            }
            
        }

        return false;
    }
    
    /**
     * @param seg0
     * @param seg1
     * @param tolerance
     * @return collect all intersection between given collection of segments. If there is no intersection,
     * then it returns empty LinkedList.
     */
    public static Collection<Latlon> collect(Collection<GeoSegment> seg0, Collection<GeoSegment> seg1, DistanceNode tolerance) {
        Collection<Latlon> _ints = new LinkedList<>();

        // ~~~ newly added.
        if (!nullNorEmpty(seg0, seg1)) {
            return _ints;
        }
        
        // every segment of seg0
        for ( GeoSegment _curr : seg0 ) {

            // every segment of seg1
            for ( GeoSegment _next : seg1 ) {

                // calculate intersection coordinate
                final Latlon _coord = _curr.intersection(_next, tolerance);

                // if intersection exists, push that to list
                if (nonNull(_coord)) {
                    _ints.add(_coord);
                }

            }

        }

        return _ints;
    }
    
    /**
     * @param seg0
     * @param seg1
     * @return true if there is any intersection between given collection of segments.
     */
    public static boolean any(Collection<GeoSegment> seg0, Collection<GeoSegment> seg1) {
        if (!nullNorEmpty(seg0, seg1)) {
            return false;
        }

        // for every segment of seg0
        for ( GeoSegment _curr : seg0 ) {

            // for every segment of seg1
            for ( GeoSegment _next : seg1 ) {

                // determine any intersection exists.
                if (_curr.intersects(_next)) {
                    return true;
                }

            }

        }

        return false;
    }
    
    /**
     *
     * @param collection
     * @return true neither null nor empty. Otherwise false.
     */
    private static boolean nullNorEmpty(Collection<?> collection) {
        if (isNull(collection)) {
            return false;
        }

        return !collection.isEmpty();
    }

    /**
     *
     * @param collection0
     * @param collection1
     * @return true neither null nor empty. Otherwise false.
     */
    private static boolean nullNorEmpty(Collection<?> collection0, Collection<?> collection1) {
        return nullNorEmpty(collection0) && nullNorEmpty(collection1);
    }
}
