/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.toRadians;
import static terraml.commons.Doubles.isEqual;
import terraml.commons.Objects;
import terraml.commons.annotation.Development;
import terraml.commons.math.Angle;
import terraml.commons.math.Vec3d;
import terraml.geospatial.impl.ImmutableLatlon;

// İhtiyaçtan fazlasında gözün olmasın
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 2.0.0-SNAPSHOT
 * @since 1.0.0-SNAPSHOT
 */
public final class GeoVector {

    public final double x;
    public final double y;
    public final double z;

    /**
     * @param double
     * @param double
     * @param double
     */
    public GeoVector(double x, double y, double z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }

    /**
     * @param Vec3d
     */
    public GeoVector(Vec3d vec3d) {
        this(vec3d.x, vec3d.y, vec3d.z);
    }

    /**
     * @param GeoVector
     */
    public GeoVector(GeoVector geoVector) {
        this(geoVector.x, geoVector.y, geoVector.z);
    }

    /**
     * @param GeoVector
     * @param Angle
     * @return
     */
    @Development(
            status = Development.Status.UNKNOWN,
            content = "Not tested yet properly."
    )
    public GeoVector offset(GeoVector _axis, Angle thetaAngle) {

        GeoVector _v0 = normalize();
        double[] _axisVecAsArr = new double[]{_v0.x, _v0.y, _v0.z};

        GeoVector _v1 = _axis.normalize();
        double _sinTheta = thetaAngle.sin();
        double _costTheta = thetaAngle.cos();

        double[][] _qdMatrix = new double[][]{
            {_v1.x * _v1.x * (1 - _costTheta) + _costTheta, _v1.x * _v1.y * (1 - _costTheta) - _v1.z * _sinTheta, _v1.x * _v1.z * (1 - _costTheta) + _v1.y * _sinTheta},
            {_v1.y * _v1.x * (1 - _costTheta) + _v1.z * _sinTheta, _v1.y * _v1.y * (1 - _costTheta) + _costTheta, _v1.y * _v1.z * (1 - _costTheta) - _v1.x * _sinTheta},
            {_v1.z * _v1.x * (1 - _costTheta) - _v1.y * _sinTheta, _v1.z * _v1.y * (1 - _costTheta) + _v1.x * _sinTheta, _v1.z * _v1.z * (1 - _costTheta) + _costTheta},};

        double[] _solution = new double[]{0, 0, 0};
        for ( int i = 0; i < 3; i++ ) {
            for ( int j = 0; j < 3; j++ ) {
                _solution[i] += _qdMatrix[i][j] * _axisVecAsArr[j];
            }
        }

        return new GeoVector(_solution[0], _solution[1], _solution[2]);
    }

    /**
     * @return
     */
    @Development(status = Development.Status.STABLE)
    public Latlon toLatlon() {
        /**
         * final double _let0 = hypot(x, y);
         * <p>
         * final double _letQ = atan2(z, _let0);
         * final double _letW = atan2(y, x);
         * <p>
         * return new ImmutableLatlon(toDegrees(_letQ), toDegrees(_letW));
         */

        double lat = atan2(this.z, Math.sqrt(this.x * this.x + this.y * this.y));
        double lon = atan2(this.y, this.x);

        return new ImmutableLatlon(Math.toDegrees(lat), Math.toDegrees(lon));
    }

    /**
     * @param Latlon
     * @return
     */
    public static GeoVector fromLatlon(Latlon latlon) {
        return fromRadianLatlon(latlon.getLatitude().toRadian(), latlon.getLongitude().toRadian());
    }

    /**
     * @param double
     * @param double
     * @return
     */
    public static GeoVector fromRadianLatlon(double lat0, double lon0) {
        final double _letX = cos(lat0) * cos(lon0);
        final double _letY = cos(lat0) * sin(lon0);
        final double _letZ = sin(lat0);

        return new GeoVector(_letX, _letY, _letZ);
    }

    /**
     * @param GeoVector
     * @param double
     * @return
     */
    private static GeoVector _getGreatCircle(GeoVector geoVector, double radianBearing) {
        final GeoVector _tmp = new GeoVector(0, 0, 1);

        GeoVector _easting = _tmp.cross(geoVector);
        GeoVector _northing = geoVector.cross(_easting);

        GeoVector _v0 = _easting.scale(cos(radianBearing) / _easting.norm());
        GeoVector _v1 = _northing.scale(sin(radianBearing) / _northing.norm());

        return _v1.translate(_v0.reverse());
    }

    /**
     * @param Angle
     * @return
     */
    public GeoVector getGreatCircle(Angle angle) {
        return new GeoVector(_getGreatCircle(this, angle.radian));
    }

    /**
     * @param double
     * @return
     */
    public GeoVector getGreatCircle(double degree) {
        return new GeoVector(_getGreatCircle(this, toRadians(degree)));
    }

    /**
     * @param GeoVector
     * @param GeoVector
     * @return
     */
    public Angle angleTo(GeoVector _v0, GeoVector _v1) {
        double _sinQ = cross(_v0).norm();
        double _cosQ = dot(_v0);

        if (!Objects.isNull(_v1)) {
            _sinQ = cross(_v0).dot(_v1) < 0 ? -_sinQ : _sinQ;
        }

        return Angle.fromRadian(atan2(_sinQ, _cosQ));
    }

    /**
     * @param double
     * @param double
     * @param double
     * @return
     */
    public GeoVector translate(double _x, double _y, double _z) {
        return new GeoVector(x + _x, y + _y, z + _z);
    }

    /**
     * @param GeoVector
     * @return
     */
    public GeoVector translate(GeoVector geoVector) {
        return translate(geoVector.x, geoVector.y, geoVector.z);
    }

    /**
     * @param double
     * @return
     */
    public GeoVector scale(double scalar) {
        return new GeoVector(x * scalar, y * scalar, z * scalar);
    }

    /**
     * @return
     */
    public GeoVector reverse() {
        return scale(-1);
    }

    /**
     * @param GeoVector
     * @return
     */
    public double dot(GeoVector geoVector) {
        final double dx = x * geoVector.x;
        final double dy = y * geoVector.y;
        final double dz = z * geoVector.z;

        return dx + dy + dz;
    }

    /**
     * @param GeoVector
     * @return
     */
    public GeoVector cross(GeoVector geoVector) {
        final double dx = y * geoVector.z - z * geoVector.getY();
        final double dy = z * geoVector.x - x * geoVector.getZ();
        final double dz = x * geoVector.y - y * geoVector.getX();

        return new GeoVector(dx, dy, dz);
    }

    /**
     * @return
     */
    public double normSq() {
        return dot(this);
    }

    /**
     * @return
     */
    public double norm() {
        return sqrt(normSq());
    }

    /**
     * @return
     */
    public GeoVector normalize() {
        final double normal = norm();
        if (isEqual(normal, 0d)) {
            throw new IllegalStateException("1/0 unknown");
        }

        return scale(1 / normal);
    }

    /**
     * @param GeoVector
     * @return
     */
    public double distSq(GeoVector geoVector) {
        return geoVector.translate(this.reverse()).normSq();
    }

    /**
     * @param GeoVector
     * @return
     */
    public double dist(GeoVector geoVector) {
        return sqrt(distSq(geoVector));
    }

    /**
     * @return
     */
    public double getX() {
        return x;
    }

    /**
     * @return
     */
    public double getY() {
        return y;
    }

    /**
     * @return
     */
    public double getZ() {
        return z;
    }
}
