/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import java.io.Serializable;
import java.util.Objects;
import terraml.commons.annotation.Development;
import terraml.commons.annotation.File;
import terraml.commons.unit.DistanceUnit;

// Beni delirtmeyin sokakları geri verin.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "DistanceNode.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
@Development(status = Development.Status.STABLE)
public final class DistanceNode implements Serializable {

    public final DistanceUnit unit;
    public final double distance;

    /**
     * @param DistanceUnit
     * @param double
     */
    public DistanceNode(DistanceUnit unit, double distance) {
        this.unit = unit;
        this.distance = distance;
    }

    /**
     * @param unit
     * @param distance
     * @return
     */
    public static DistanceNode of(DistanceUnit unit, double distance) {
        return new DistanceNode(unit, distance);
    }

    /**
     * @param distance
     * @return
     */
    public static DistanceNode fromMeter(double distance) {
        return new DistanceNode(DistanceUnit.METER, distance);
    }

    /**
     * @param distance
     * @return
     */
    public static DistanceNode fromKilometer(double distance) {
        return new DistanceNode(DistanceUnit.KILOMETER, distance);
    }

    /**
     * @return
     */
    public double asMeter() {
        if (unit.equals(DistanceUnit.METER)) {
            return distance;
        }

        return DistanceUnit.KILOMETER.toMeter(distance);
    }

    /**
     * @return
     */
    public double asKilometer() {
        if (unit.equals(DistanceUnit.KILOMETER)) {
            return distance;
        }

        return DistanceUnit.METER.toKilometer(distance);
    }

    /**
     * @return
     */
    public DistanceUnit getUnit() {
        return unit;
    }

    /**
     * @return
     */
    public double getDistance() {
        return distance;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + Objects.hashCode(this.unit);
        hash = 29 * hash + (int) (Double.doubleToLongBits(this.distance) ^ (Double.doubleToLongBits(this.distance) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final DistanceNode other = (DistanceNode) obj;
        if (Double.doubleToLongBits(this.distance) != Double.doubleToLongBits(other.distance)) {
            return false;
        }
        if (this.unit != other.unit) {
            return false;
        }
        return true;
    }

    // herkes delirmiş.
}
