/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.abs;
import static java.lang.Math.acos;
import static java.lang.Math.atan;
import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.tan;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.annotation.Development;
import terraml.commons.annotation.File;
import terraml.commons.annotation.Reference;
import terraml.commons.unit.DistanceUnit;
import static terraml.geospatial.GeoUtils.lat2rad;
import static terraml.geospatial.GeoUtils.lon2rad;

// ateş edelim istersen mafyaya.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "Distance.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
@Development(status = Development.Status.STABLE)
public final class Distance {

    private Distance() {
    }

    /**
     *
     * final DistanceCalculator calc = new Vincenty();
     * final double dist = calc.distanceOf(new Latlon(0,0), new Latlon(0,0));
     * <p>
     */
    public static class Vincenty implements DistanceCalculator {

        @Override
        public DistanceNode distanceOf(Latlon source, Latlon target) {
            return new DistanceNode(DistanceUnit.METER, vincenty(source, target));
        }

    }

    /**
     *
     * final DistanceCalculator calc = new Haversine();
     * final double dist = calc.distanceOf(new Latlon(0,0), new Latlon(0,0));
     * <p>
     */
    public static class Haversine implements DistanceCalculator {

        @Override
        public DistanceNode distanceOf(Latlon source, Latlon target) {
            return new DistanceNode(DistanceUnit.METER, haversine(source, target));
        }

    }

    /**
     *
     * final DistanceCalculator calc = new LawOfCosine();
     * final double dist = calc.distanceOf(new Latlon(0,0), new Latlon(0,0));
     * <p>
     */
    public static class LawOfCosine implements DistanceCalculator {

        @Override
        public DistanceNode distanceOf(Latlon source, Latlon target) {
            return new DistanceNode(DistanceUnit.METER, lawOfCosine(source, target));
        }

    }

    /**
     *
     * @param lat1
     * @param lon1
     * @param lat2
     * @param lon2
     * @return
     */
    @Reference(
            title = "Vincenty Formula",
            link = "https://en.wikipedia.org/wiki/Vincenty%27s_formulae"
    )
    public static double vincentyFromRadian(double lat1, double lon1, double lat2, double lon2) {
        double a = 6378137, b = 6356752.314245, f = 1 / 298.257223563;
        double L = lon2 - lon1;
        double U1 = atan((1 - f) * tan(lat1));
        double U2 = atan((1 - f) * tan(lat2));
        double sinU1 = sin(U1), cosU1 = cos(U1);
        double sinU2 = sin(U2), cosU2 = cos(U2);

        double sinLambda, cosLambda, sinSigma, cosSigma, sigma, sinAlpha, cosSqAlpha, cos2SigmaM;
        double lambda = L, lambdaP, iterLimit = 100;
        do {
            sinLambda = sin(lambda);
            cosLambda = cos(lambda);
            sinSigma = sqrt((cosU2 * sinLambda) * (cosU2 * sinLambda)
                    + (cosU1 * sinU2 - sinU1 * cosU2 * cosLambda) * (cosU1 * sinU2 - sinU1 * cosU2 * cosLambda));
            if (sinSigma == 0) {
                return 0;
            }
            cosSigma = sinU1 * sinU2 + cosU1 * cosU2 * cosLambda;
            sigma = atan2(sinSigma, cosSigma);
            sinAlpha = cosU1 * cosU2 * sinLambda / sinSigma;
            cosSqAlpha = 1 - sinAlpha * sinAlpha;
            cos2SigmaM = cosSigma - 2 * sinU1 * sinU2 / cosSqAlpha;
            if (Double.isNaN(cos2SigmaM)) {
                cos2SigmaM = 0;
            }
            double C = f / 16 * cosSqAlpha * (4 + f * (4 - 3 * cosSqAlpha));
            lambdaP = lambda;
            lambda = L + (1 - C) * f * sinAlpha
                    * (sigma + C * sinSigma * (cos2SigmaM + C * cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM)));
        } while ( abs(lambda - lambdaP) > 1e-12 && --iterLimit > 0 );

        if (iterLimit == 0) {
            return Double.NaN;
        }
        double uSq = cosSqAlpha * (a * a - b * b) / (b * b);
        double A = 1 + uSq / 16384 * (4096 + uSq * (-768 + uSq * (320 - 175 * uSq)));
        double B = uSq / 1024 * (256 + uSq * (-128 + uSq * (74 - 47 * uSq)));
        double deltaSigma = B
                * sinSigma
                * (cos2SigmaM + B
                   / 4
                   * (cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM) - B / 6 * cos2SigmaM
                      * (-3 + 4 * sinSigma * sinSigma) * (-3 + 4 * cos2SigmaM * cos2SigmaM)));

        return b * A * (sigma - deltaSigma);
    }

    /**
     * Note: Use this as much as possible whenever you need to calculate distance. It's the most accurate calculation.
     *
     * @param latlon0
     * @param latlon1
     * @return
     */
    public static double vincenty(Latlon latlon0, Latlon latlon1) {
        return vincentyFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1));
    }

    /**
     *
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     * @return in meters.
     */
    @Reference(
            title = "Haversine Formula",
            link = "https://en.wikipedia.org/wiki/Haversine_formula"
    )
    public static double haversineFromRadian(double lat0, double lon0, double lat1, double lon1) {
        final double _letX = lat1 - lat0;
        final double lat = sin(0.5 * _letX);

        final double _letY = lon1 - lon0;
        final double lon = sin(0.5 * _letY);

        double _let0 = (lon * lon) * cos(lat0) * cos(lat1);
        _let0 += (lat * lat);

        double _let1 = atan2(sqrt(_let0), sqrt(1 - _let0));
        _let1 *= 2;

        return _let1 * GeoUtils.EARTH_RADIUS_M;
    }

    /**
     *
     * @param latlon0
     * @param latlon1
     * @return
     */
    public static double haversine(Latlon latlon0, Latlon latlon1) {
        return haversineFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1));
    }

    /**
     *
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     * @return
     */
    @Reference(
            title = "Law of Cosine (Spherical)",
            link = "https://en.wikipedia.org/wiki/Law_of_cosines"
    )
    public static double lawOfCosineFromRadian(double lat0, double lon0, double lat1, double lon1) {
        if (isEqual(lat0, lat1) && isEqual(lon0, lon1)) {
            return 0.0d;
        }

        final double _letY = lon1 - lon0;
        final double _let0 = (Math.PI / 2) - lat0;
        final double _let1 = (Math.PI / 2) - lat1;
        final double _let2 = (cos(_let0) * cos(_let1)) + (sin(_let0) * sin(_let1) * cos(_letY));

        if (isSmaller(_let2, -1.0)) {
            return Math.PI;
        } else if (isGreaterEqual(_let2, 1.0)) {
            return 0.0d;
        }

        return acos(_let2);
    }

    /**
     *
     * @param latlon0
     * @param latlon1
     * @return
     */
    public static double lawOfCosine(Latlon latlon0, Latlon latlon1) {
        return lawOfCosineFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1));
    }
}
