/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.abs;
import static java.lang.Math.asin;
import static java.lang.Math.sin;
import terraml.commons.annotation.File;
import terraml.commons.unit.DistanceUnit;
import static terraml.geospatial.Azimuths.northBasedAzimuthFromRadian;
import static terraml.geospatial.GeoUtils.EARTH_RADIUS_M;

// bir maganda kurşunusun ya da bir köşede gasp.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "Crosstracks.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
public final class Crosstracks {

    private Crosstracks() {
    }

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     * @param double
     * @param double
     * @return Calculates crosstrack Haversine distance from given arguments in meter.
     */
    public static DistanceNode crossTrackHavDistanceFromRadian(double lat0, double lon0, double lat1, double lon1, double lat2, double lon2) {
        final double _let0 = Distance.haversineFromRadian(lat1, lon1, lat2, lon2) / EARTH_RADIUS_M;
        final double _let1 = northBasedAzimuthFromRadian(lat1, lon1, lat0, lon0).radian;
        final double _let2 = northBasedAzimuthFromRadian(lat1, lon1, lat2, lon2).radian;

        return new DistanceNode(DistanceUnit.METER, abs(asin(sin(_let0) * sin(_let1 - _let2)) * EARTH_RADIUS_M));
    }

    /**
     * @param double[]
     *                 @
     * param double[]
     *                @
     * param double[]
     *                @
     * return Calculates crosstrack Haversine distance from given arguments in meter.
     */
    public static DistanceNode crossTrackHavDistanceFromRadian(double[] latlon, double[] source, double[] target) {
        return crossTrackHavDistanceFromRadian(latlon[0], latlon[1], source[0], source[1], target[0], target[1]);
    }

    /**
     * @param Latlon
     * @param Latlon
     * @param Latlon
     * @return Calculates crosstrack Haversine distance from given arguments in meter.
     */
    public static DistanceNode crossTrackHavDistance(Latlon latlon, Latlon source, Latlon target) {
        return crossTrackHavDistanceFromRadian(latlon.toArrayAsRadian(), source.toArrayAsRadian(), target.toArrayAsRadian());
    }

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     * @param double
     * @param double
     * @return Calculates crosstrack Vincenty distance from given arguments in meter.
     */
    public static DistanceNode crossTrackVinDistanceFromRadian(double lat0, double lon0, double lat1, double lon1, double lat2, double lon2) {
        final double _let0 = Distance.vincentyFromRadian(lat1, lon1, lat2, lon2) / EARTH_RADIUS_M;
        final double _let1 = northBasedAzimuthFromRadian(lat1, lon1, lat0, lon0).radian;
        final double _let2 = northBasedAzimuthFromRadian(lat1, lon1, lat2, lon2).radian;

        return new DistanceNode(DistanceUnit.METER, abs(asin(sin(_let0) * sin(_let1 - _let2)) * EARTH_RADIUS_M));
    }

    /**
     * @param double[]
     *                 @
     * param double[]
     *                @
     * param double[]
     *                @
     * return Calculates crosstrack Vincenty distance from given arguments in meter.
     */
    public static DistanceNode crossTrackVinDistanceFromRadian(double[] latlon, double[] source, double[] target) {
        return crossTrackVinDistanceFromRadian(latlon[0], latlon[1], source[0], source[1], target[0], target[1]);
    }

    /**
     * @param Latlon
     * @param Latlon
     * @param Latlon
     * @return Calculates crosstrack Vincenty distance from given arguments in meter.
     */
    public static DistanceNode crossTrackVinDistance(Latlon latlon, Latlon source, Latlon target) {
        return crossTrackVinDistanceFromRadian(latlon.toArrayAsRadian(), source.toArrayAsRadian(), target.toArrayAsRadian());
    }
}
