/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import terraml.commons.annotation.File;
import terraml.commons.math.Vec3d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Point3D;
import terraml.geometry.ShapeUnit;
import terraml.geometry.Sphere3D;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 *
 */
@File(
        fileName = "ImmutableSphere3D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableSphere3D implements Sphere3D, Serializable {

    private final Point3D _center;
    private final double _radius;

    /**
     * @param Point3D
     * @param double
     */
    public ImmutableSphere3D(Point3D _center, double _radius) {
        this._center = _center;
        this._radius = _radius;
    }

    /**
     * @param Sphere3D
     */
    public ImmutableSphere3D(Sphere3D _sphere) {
        this(_sphere.getCenter(), _sphere.getRadius());
    }

    @Override
    public Vec3d getNormal(Point3D point3D) {
        final Vec3d _v0 = _center.toVector();
        final Vec3d _v1 = point3D.toVector();

        return _v1.sub(_v0).getNormalized();
    }

    @Override
    public double getDiameter() {
        return 2.0 * _radius;
    }

    @Override
    public double getVolume() {
        return (4.0 / 3.0) * Math.PI * _radius * _radius * _radius;
    }

    @Override
    public List<Point3D> getBounds() {
        final Vec3d _v0 = _center.toVector();
        final Vec3d _v1 = new Vec3d(_radius, _radius, _radius);
        final Vec3d _v2 = _v0.sub(_v1);
        final Vec3d _v3 = _v0.add(_v1);

        final Point3D _lowB = new ImmutablePoint3D(_v2.x, _v2.y, _v2.z);
        final Point3D _uppB = new ImmutablePoint3D(_v3.x, _v3.y, _v3.z);

        return Arrays.asList(_lowB, _uppB);
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public Point3D getCenter() {
        return this._center;
    }

    @Override
    public double getRadius() {
        return this._radius;
    }

    @Override
    public Sphere3D copy() {
        return new ImmutableSphere3D(_center, _radius);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.THREE;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.SPHERE;
    }

    @Override
    public Sphere3D translate(double... args) {
        final Point3D p = _center.translate(args);

        return new ImmutableSphere3D(p, _radius);
    }

    @Override
    public Sphere3D scale(double scaleFactor) {
        final Point3D p = _center.scale(scaleFactor);

        return new ImmutableSphere3D(p, _radius);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 37 * hash + Objects.hashCode(this._center);
        hash = 37 * hash + (int) (Double.doubleToLongBits(this._radius) ^ (Double.doubleToLongBits(this._radius) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableSphere3D other = (ImmutableSphere3D) obj;
        if (Double.doubleToLongBits(this._radius) != Double.doubleToLongBits(other._radius)) {
            return false;
        }
        if (!Objects.equals(this._center, other._center)) {
            return false;
        }
        return true;
    }
}
