/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geometry project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geometry.impl;

import java.io.Serializable;
import static java.lang.Math.max;
import static java.lang.Math.min;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.annotation.File;
import terraml.commons.math.Interval;
import terraml.commons.math.Vec3d;
import terraml.commons.unit.DimensionUnit;
import terraml.geometry.Line3D;
import terraml.geometry.Point3D;
import terraml.geometry.Segment3D;
import terraml.geometry.ShapeUnit;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "ImmutableSegment3D.java",
        packageName = "terraml.geometry.impl",
        projectName = "terraml-geometry"
)
public class ImmutableSegment3D implements Segment3D, Serializable {

    private final Point3D _source;
    private final Point3D _target;

    /**
     * @param Point3D
     * @param Point3D
     */
    public ImmutableSegment3D(Point3D source, Point3D target) {
        this._source = source;
        this._target = target;
    }

    /**
     * @param Segment3D
     */
    public ImmutableSegment3D(Segment3D segment3D) {
        this(segment3D.getSource(), segment3D.getTarget());
    }

    /**
     * @param Segment3D
     * @param Point3D
     * @return
     */
    protected Point3D closestPoint(Segment3D _s0, Point3D _p0) {
        final Vec3d _v0 = _s0.getSource().toVector();
        final Vec3d _v1 = _s0.getTarget().toVector();
        final Vec3d _v2 = _p0.toVector();
        final Vec3d _v3 = _v1.sub(_v0);
        final Vec3d _v4 = _v2.sub(_v0);

        final double _let0 = _v4.getDotProduct(_v3);
        final double _let1 = _v3.getDotProduct(_v3);

        if (isSmallerEqual(_let0, 0)) {
            return _s0.getSource();
        }

        if (isSmallerEqual(_let1, _let0)) {
            return _s0.getTarget();
        }

        final Vec3d _v5 = _v3.scale(_let0 / _let1);
        final Vec3d _v6 = new Vec3d(_v5.getX(), _v5.getY(), _v5.getZ());
        final Vec3d _v7 = _v0.add(_v6);

        return new ImmutablePoint3D(_v7.x, _v7.y, _v7.z);
    }

    @Override
    public Point3D closestPoint(Point3D point3D) {
        return closestPoint(this, point3D);
    }

    @Override
    public double project(Point3D point3D) {
        final Vec3d _src = _source.toVector();
        final Vec3d _dst = _target.toVector();

        final Vec3d _dirr = _dst.sub(_src);
        final Line3D _line0 = new ImmutableLine3D(_source, _dirr);
        final double _proj = _line0.project(point3D);

        return min(max(_proj, 0d), 1);
    }

    @Override
    public Point3D point(double scalar) {
        final double _let0 = max(min(scalar, 1), 0);

        final Vec3d _src = _source.toVector();
        final Vec3d _dst = _target.toVector();
        final Vec3d _v0 = _dst.sub(_src);

        final double _x0 = _source.getX() + _v0.x * _let0;
        final double _y0 = _source.getY() + _v0.y * _let0;
        final double _z0 = _source.getZ() + _v0.z * _let0;

        return new ImmutablePoint3D(_x0, _y0, _z0);
    }

    // calculation might be wrong, check later. Try _distanceTo2 instead of this.
    @Override
    public double distanceTo(Point3D point3D) {
        if (_source.equals(_target)) {
            return _source.distanceTo(point3D);
        }

        final Vec3d _src = _source.toVector();
        final Vec3d _dst = _target.toVector();
        final Vec3d _ptr = point3D.toVector();

        final Vec3d _v0 = _dst.sub(_src);
        final Vec3d _v1 = _ptr.sub(_src);

        final double _let0 = _v1.getDotProduct(_v0);
        if (isSmallerEqual(_let0, 0d)) {
            return _v1.getNorm();
        }

        final Vec3d _v2 = _ptr.sub(_dst);
        if (isGreaterEqual(_v2.getDotProduct(_v1), 0)) {
            return _v2.getNorm();
        }

        return _v0.getCrossProduct(_v1).getNorm() / _v0.getNorm();
    }

    /**
     * @param Point3D
     * @return
     */
    private double _distanceTo2(Point3D point3D) {
        final double _let0 = project(point3D);

        return point(_let0).distanceTo(point3D);
    }

    @Override
    public Vec3d getDirection() {
        final Vec3d _src = _source.toVector();
        final Vec3d _dst = _target.toVector();

        return _dst.sub(_src).getNormalized();
    }

    @Override
    public double getLength() {
        return _source.distanceTo(_target);
    }

    @Override
    public Point3D getSource() {
        return this._source;
    }

    @Override
    public Point3D getTarget() {
        return this._target;
    }

    @Override
    public boolean isHorizontal() {
        return isEqual(_source.getY(), _target.getY());
    }

    @Override
    public boolean isVertical() {
        return isEqual(_source.getX(), _target.getX());
    }

    @Override
    public List<Point3D> getBounds() {
        final Interval _xBound = new Interval(_source.getX(), _target.getX());
        final Interval _yBound = new Interval(_source.getY(), _target.getY());
        final Interval _zBound = new Interval(_source.getZ(), _target.getZ());

        final Point3D _lowerBounding = new ImmutablePoint3D(_xBound.left, _yBound.left, _zBound.left);
        final Point3D _upperBounding = new ImmutablePoint3D(_xBound.right, _yBound.right, _zBound.right);

        return Arrays.asList(_lowerBounding, _upperBounding);
    }

    @Override
    public boolean isBounded() {
        return true;
    }

    @Override
    public ImmutableSegment3D copy() {
        return new ImmutableSegment3D(_source, _target);
    }

    @Override
    public DimensionUnit getDimensionUnit() {
        return DimensionUnit.THREE;
    }

    @Override
    public ShapeUnit getShapeUnit() {
        return ShapeUnit.SEGMENT;
    }

    @Override
    public Segment3D translate(double... args) {
        final Point3D p0 = getSource().translate(args);
        final Point3D p1 = getTarget().translate(args);

        return new ImmutableSegment3D(p0, p1);
    }

    @Override
    public Segment3D scale(double scaleFactor) {
        final Point3D p0 = getSource().scale(scaleFactor);
        final Point3D p1 = getTarget().scale(scaleFactor);

        return new ImmutableSegment3D(p0, p1);
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 29 * hash + Objects.hashCode(this._source);
        hash = 29 * hash + Objects.hashCode(this._target);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableSegment3D other = (ImmutableSegment3D) obj;
        if (!Objects.equals(this._source, other._source)) {
            return false;
        }
        if (!Objects.equals(this._target, other._target)) {
            return false;
        }
        return true;
    }
}
